import { defaultLocale, localeCacheKey, localeDecoder, locales } from '../../../config';
import * as eff from '../../eff';
import { Ctx as TranslationCtx, Translations } from '../../gettext';
import { Cache as LocalStorage } from '../../persistent';
import * as bcp47 from '../../utils/bcp47';
import { Async, LocalStorageError, TranslationLoadError } from '../error';
export const defaultTranslations: Translations = { "locale_data" : { "messages" : { "" : { "domain": "messages", "lang": "en", "plural_forms" : "nplurals=2; plural=(n != 1);" } } }, "domain" : "messages", "debug" : false }

export type Locale = typeof localeDecoder['_A'];
export type Locales = Locale[];

export const cache = new LocalStorage<Locale>(localeCacheKey, localeDecoder);
/**
 * Возвращает в случаи успеха `Translations`. Необходим для работы с `gettext`
 * @param locale вариант перевода из `Locales`
 */
export function loadTranslations(locale: Locale): Async<Translations> {
  try {
    const translations: Translations | undefined = require(`../../../i18n/${locale}.po`);
    return translations ? eff.success(translations) : eff.failure(TranslationLoadError(locale));
  } catch(e) {
    return eff.of(defaultTranslations);
  }
}
/**
 * Возвращает подходящую локаль из `window.navigator.languages` или `defaultLocale`
 */
export function BrowserLocale(): Locale {
  return (window.navigator.languages ? window.navigator.languages.slice() : [])
    .map((locale) => bcp47.lookupLocale(locale, locales)) // Приводит к BCP47, если это возможно
    .filter(browser => locales.find(locale => locale === browser))[0]
  || defaultLocale;
}
/**
 * Контекст для Locale
 */
export type Ctx = TranslationCtx & { locale: Locale };
/**
 * Инициализация локализации приложения по следущей логике:
 * Если в `LocalStorage` присутсвует установленная локаль, подгружает ее.
 * В случаи если `LocalStorage` значение отсуствует, выбор предоставляется из `локали браузера`
 * @param locale вариант перевода из `Locales`
 */
export function init(): Async<Ctx> {
  return cache.read().onError(e => {
    switch (e.tag) {
      case 'NoItem': return eff.success(BrowserLocale());
      default: return eff.failure(LocalStorageError(e));
    }
  }).chain((locale) => loadTranslations(locale)
    .map(translations => ({ translations, locale })),
  );
}
