# Соглашение по именованию методов

Эта библиотека содержит модули [eff](eff), [decoder](decoder), [either](either),
[maybe](maybe) которые экспортируют статические и инстанс-методы
названия которых имитируют названия из функциональных языков. Эти
модули часто используются в неокторых приложениях (Кабинет Оператора,
ЕЛК, GuestGuide, Кабинет Аналитика)

### Functor
```ts
interface Functor<F> {
    map<A>(this: F<A>, proj: (a: A) => B): F<B>;
    mapTo<A>(this: F<A>, value: B): F<B>;
}
```

`map` делает тоже самое что и `Array.prototype.map` но для любого `F`,
пример —
[Eff.ptototype.map](https://git.bitmaster.ru/npm/common/blob/218d38e2247ae49c5b3891007a118e6eace9316c/eff/index.ts#L22),
[Decoder.prototype.map](https://git.bitmaster.ru/npm/common/blob/218d38e2247ae49c5b3891007a118e6eace9316c/decoder/index.ts#L93)
`mapTo` это версия `map` в которой игнорируется аргумент внутри `proj`

Дополнительная информация по интерфейсу `Functor`
[https://wiki.haskell.org/Functor](https://wiki.haskell.org/Functor)

```
this.mapTo(value) = this.map(() => value)
```

### Applicative
```ts
interface Applicative<F> {
    of<A>(value: A): F<A>;

    ap<A,B>(a: Eff<L,A>, f: (a: A) => B): Eff<L,B>;
    ap<A,B,C>(a: Eff<L,A>, b: Eff<L,B>, f: (a: A, b: B) => C): Eff<L,C>;
    
    traverse<A, B>(array: A[], f: (a: A, idx: number) => F<B>): F<B[]>;
    
    record<T>(fields: { [K in keyof T]: F<T[K]> }): F<T>;
    tuple<A extends F<any>[]>(...reps: A): F<{ [K in keyof A]: F['_A'] }>;
}
```

`of` это универсальный конструктор для типа `F` если проводить
аналогию с известными типами `Array` и `Promise` то 
```
of = value => [value] // для Array
of = Promise.resolve // для Promise
```

`ap` можно считать что это `map` но для любого кол-ва `this`

`traverse` если проводить аналогию с `Promise` то

```
traverse = (xs, f) => Promise.all(xs.map(f));
``` 

`record` и `tuple` можно считать шорткатами для `ap` которые удобнее использовать в некоторых случаях
```

tuple = (...xs) => ap(...xs, (...xs) => xs);

record = (rec) => { 
  const keys = Object.keys(rec);
  retrn ap(...keys.map(k => rec[k]), values => value.reduce((acc, v, idx) => (acc[keys[idx]], acc), {}));
};
```
Дополнительная информация по интерфейсу `Applicative`
[https://wiki.haskell.org/Applicative_functor](https://wiki.haskell.org/Applicative_functor)


### Monad

```ts
interface Monad<F> {
    chain<A, B>(this: F<A>, then: (a: A) => F<B>): F<B>;
    chainTo<A, B>(this: F<A>, value: F<B>): F<B>;
}
```

`chain` похож на `Promise.prototype.then` но определен для любого `F`
который релизут `Monad<F>`
`chainTo` это версия `chain` в которой игнорируется аргумент внутри `proj`

Дополнительная информация по интерфейсу `Monad`
[https://wiki.haskell.org/Monad](https://wiki.haskell.org/Monad)

### Contravariant

```ts
interface Contravariant<F> {
  comap<A>(this: F<A>, coproj: (a: B) => A): F<B>;
}
```

`comap` похож на `map' но аргумент и результат в coproj поменяны местами

Дополнительная информация
[https://en.wikipedia.org/wiki/Covariance_and_contravariance_(computer_science)](https://en.wikipedia.org/wiki/Covariance_and_contravariance_(computer_science))

### Profunctor

```ts
interface Profunctor<F> {
  dimap<A, B, C, D>(this: F<B, C>, coproj: (a: A) => B, proj: (c: C) => D): F<A, D>;
}
```

`dimap` похож на `map', но он принимает две функции как аргумент, 

Дополнительная информация по интерфейсу `Profunctor`
[https://en.wikipedia.org/wiki/Profunctor](https://en.wikipedia.org/wiki/Profunctor)
