import * as React from 'react';
import { Eff } from '../eff';
import { Either } from '../either';
import { absurd } from '../types';


// ADT
export type Update<State, A> =
  | Pure<State, A>     // { value: A }
  | Modify<State, A>   // { proj(x: State): State } // A ~ void
  | Patch<State, A>    // { patch: Partial<State> } // A ~ void
  | Get<State, A>      // { tag: 'Get' } // A ~ State
  | Children<State, A> // { key: ControlledComponent<any, any>, update: Update<any, A> }
  | Effect<State, A>   // { effect: Eff<never, A> }
  | FromPromise<State, A> // { promise: Promise<A> }
  | Chain<State, A>    // { update: Update<State, any>, chain(x: any): Update<State, A> }
  | Apply<State, A>    // { args: Update<State, any>[], proj(...args): A }
  | Batch<State, A>    // { steps: Updte<State, any>[] }
  | Concat<State, A>   // { steps: Updte<State, any>[] }
  ;


// Perform state update
export function go<State, A>(update: Update<State, A>, component: React.Component<any, State>, onNext: (x: A) => void, onComplete: () => void): () => void {
  if (update instanceof Pure) {
    onNext(update.value);
    onComplete();
    return noopFunc;
  }
  
  if (update instanceof Modify) {
    component['updater'].enqueueSetState(component, update.proj, () => {
      onNext(void 0 as any);
      update.callback && update.callback();
      onComplete();
    });
    return noopFunc;
  }
  
  if (update instanceof Patch) {
    component['updater'].enqueueSetState(component, update.patch, () => {
      onNext(void 0 as any);
      update.callback && update.callback();
      onComplete();
    });
    return noopFunc;
  }
  
  if (update instanceof Get) {
    onNext(component.state as any);
    onComplete();
    return noopFunc;
  }
  
  if (update instanceof Children) {
    return go(update.update, update.component, onNext, onComplete);
  }
  
  if (update instanceof Effect) {
    return update.effect.run(onNext as any, onComplete);
  }

  if (update instanceof FromPromise) {
    update.promise().then(x => (onNext(x), onComplete()));
    return noopFunc;
  }
  
  if (update instanceof Chain) {
    const subscriptions: Array<Function|null> = [];
    subscriptions.push(go(update.update, component, result => {
      const idx = subscriptions.length;
      subscriptions.push(go(update.andThen(result), component, onNext, () => {
        subscriptions[idx] = null;
        for (const unsub of subscriptions) if (unsub !== null) return;
        onComplete();
      }));
    }, () => {
      subscriptions[0] = null;
      for (const unsub of subscriptions) if (unsub !== null) return;
      onComplete();
    }));
    
    return () => subscriptions.forEach(
      funOrNull => funOrNull ? funOrNull() : void 0
    );
  }
  
  if (update instanceof Apply) {
    let allInitialized = false;
    let subscriptions: Array<Function|undefined|null> = new Array(update.args.length);
    const initializedFlags: Array<true|undefined> = new Array(update.args.length);
    const recentValues: unknown[] = new Array(update.args.length);
    const next = idx => result => {
      recentValues[idx] = result;
      check_initialized: {
        if (allInitialized) break check_initialized;
        initializedFlags[idx] = true;
        for (const flag of initializedFlags) if (flag !== true) return;
        allInitialized = true;
      }
      onNext(update.proj.apply(void 0, recentValues));
    };
    const complete = idx => () => {
      subscriptions[idx] = null;
      for (const unsub of subscriptions) if (unsub !== null) return;
      onComplete();
    };      

    update.args.forEach((u, idx) => {
      const canceller = go(u, component, next(idx), complete(idx));
      if (subscriptions[idx] !== null) subscriptions[idx] = canceller;
    });

    return () => subscriptions.forEach(
      funOrNull => funOrNull ? funOrNull() : void 0
    );
  }
  
  if (update instanceof Batch) {
    if (update.steps.length === 0) { onNext(void 0 as any); onComplete(); return noopFunc; }
    let subscriptions: Array<Function|null>;
    const loop = idx => () => {
      subscriptions[idx] = null;
      for (const unsub of subscriptions) if (unsub !== null) return;
      onNext(void 0 as any);
      onComplete(); // If control flow reaches here, that means all nested commands are completed
    };
    subscriptions = update.steps.map((u, idx) => go(u, component, noopFunc, loop(idx)));
    
    return () => subscriptions.forEach(
      funOrNull => funOrNull ? funOrNull() : void 0
    );
  }

  if (update instanceof Concat) {
    let unsubscribe: Function|null = null;
    const loop = idx => () => {
      // If condition holds, then all nested effects are completed, therefore we're done
      if (idx >= update.steps.length) { onNext(void 0 as any); onComplete(); return; }
      unsubscribe = go(update.steps[idx], component, noopFunc, loop(idx + 1));
    };
    loop(0);
    return () => unsubscribe ? unsubscribe() : void 0;
  }

  return absurd(update);
}


// Instance methods
export class UpdateBase<State, A> {
  readonly _State: State;
  readonly _A: A;
  
  map<B>(proj: (x: A) => B): Apply<State, B> {
    const self = this as any as Update<State, A>;
    return new Apply<State, B>([self], proj);
  }

  mapTo<B>(value: B): Apply<State, B> {
    const self = this as any as Update<State, A>;
    return new Apply<State, B>([self], () => value);
  }
  
  chain<B,State2>(andThen: (x: A) => Update<State2, B>): Chain<State&State2, B>;
  chain<B>(andThen: (x: A) => Update<State, B>): Chain<State, B>;
  chain<B>(andThen: (x: A) => Update<State, B>): Chain<State, B> {
    const self = this as any as Update<State, A>;
    return new Chain<State, B>(self, andThen);
  }

  chainTo<B,State2>(value: Update<State2, B>): Chain<State&State2, B>;
  chainTo<B>(value: Update<State, B>): Chain<State, B>;
  chainTo<B>(value: Update<State, B>): Chain<State, B> {
    const self = this as any as Update<State, A>;
    return new Chain<State, B>(self, () => value);
  }

  run(component: React.Component<any, State>, onNext?: (x: A) => void, onComplete?: () => void): () => void {
    const self = this as any as Update<State, A>;
    return go(self, component, onNext || noopFunc, onComplete || noopFunc);
  }

  pending(): Update<State & { pending: boolean }, A> {
    // @ts-ignore
    return patch({ pending: true }).chainTo(this.chain(value => patch({ pending: false }).mapTo(value)));
  }
}


export function ap<State,A,B>(a: Update<State,A>, f: (a: A) => B): Update<State,B>;
export function ap<State,A,B,C>(a: Update<State,A>, b: Update<State,B>, f: (a: A, b: B) => C): Update<State,C>;
export function ap<State,A,B,C,D>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, f: (a: A, b: B, c: C) => D): Update<State,D>;
export function ap<State,A,B,C,D,E>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, f: (a: A, b: B, c: C, d: D) => E): Update<State,E>;
export function ap<State,A,B,C,D,E,F>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, e: Update<State,E>, f: (a: A, b: B, c: C, d: D, e: E) => F): Update<State,F>;
export function ap<State,A,B,C,D,E,F,G>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, e: Update<State,E>, f_: Update<State,F>, f: (a: A, b: B, c: C, d: D, e: E, f: F) => G): Update<State,G>;
export function ap(): Update<unknown, unknown> {
  const args = Array.prototype.slice.call(arguments, 0, arguments.length - 1);
  const proj = arguments[arguments.length - 1];
  return new Apply(args, proj);
}


export function of<A>(value: A): Pure<never, A> {
  return new Pure(value);
}

export function patch<State>(patch: Partial<State>, callback?: () => void): Patch<State, void> {
  return new Patch(patch, callback);
}

export function modify<State>(proj: (x: State) => State, callback?: () => void): Modify<State, void> {
  return new Modify(proj, callback);
}

export function effect<Error, Success>(eff: Eff<Error, Success>): Effect<{}, Either<Error, Success>> {
  return new Effect(eff as any);
}

export function batch<State>(...steps: Update<State, any>[]): Batch<State, void>;
export function batch<State>(steps: Update<State, any>[]): Batch<State, void>;
export function batch(): Batch<unknown, void> {
  const steps = Array.isArray(arguments[0]) ? arguments[0] : arguments;
  return new Batch(steps);
}

export function concat<State>(...steps: Update<State, any>[]): Concat<State, void>;
export function concat<State>(steps: Update<State, any>[]): Concat<State, void>;
export function concat(): Concat<unknown, void> {
  const steps = Array.isArray(arguments[0]) ? arguments[0] : arguments;
  return new Concat(steps);
}



class Pure<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly value: A,
  ) { super(); }
}

class Modify<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly proj: (x: State) => State,
    readonly callback?: () => void,
  ) { super(); }
}

class Patch<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly patch: Partial<State>,
    readonly callback?: () => void,
  ) { super(); }
}

class Get<State, A> extends UpdateBase<State, A> {
}

class Children<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly component: React.Component<any, any>,
    readonly update: Update<any, A>
  ) { super(); }
}

class Effect<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly effect: Eff<never, A>,
  ) { super(); }
}

class FromPromise<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly promise: () => Promise<A>,
  ) { super(); }
}

class Chain<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly update: Update<State, any>,
    readonly andThen: (x: any) => Update<State, A>,
  ) { super(); }
}

class Apply<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly args: Update<State, any>[],
    readonly proj: (...args) => A,
  ) { super(); }
}

class Batch<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly steps: Update<State, any>[],
  ) { super(); }
}

class Concat<State, A> extends UpdateBase<State, A> {
  constructor(
    readonly steps: Update<State, any>[],
  ) { super(); }
}


// Functional helpers
const noopFunc = () => {};


export const get = new Get();



export class Component<Props, State> extends React.Component<Props, State> {
  constructor(props, context) {
    super(props, context);
    // @ts-ignore
    if (typeof(props.initialState) !== 'undefined') this.state = props.initialState;
  }

  setState<K extends keyof State>(
    state: ((prevState: Readonly<State>, props: Readonly<Props>) => (Pick<State, K> | State | null)) | (Pick<State, K> | State | null),
    callback?: () => void
  ): void;
  setState(update: Update<State, void>, callback?: () => void): void;
  setState(updateOrProjOrPatch, callback?) {
    if (updateOrProjOrPatch instanceof UpdateBase) {
      const onUpdate = this.props['onUpdate'] as unknown;
      if (typeof(onUpdate) === 'function') {
        onUpdate(updateOrProjOrPatch);
      } else {
        updateOrProjOrPatch.run(this, callback);
      }
      return;
    }

    if (typeof(updateOrProjOrPatch) === 'function') {
      return this.setState(new Modify<State, void>(updateOrProjOrPatch, callback))
    } else {
      return this.setState(new Patch<State, void>(updateOrProjOrPatch, callback))
    }
  }
}

export interface UpdateStatic {
  ap: typeof ap,
  of: typeof of,
  patch: typeof patch,
  modify: typeof modify,
  effect: typeof effect,
  get: typeof get,
  batch: typeof batch,
  concat: typeof concat,
  Component: typeof Component,
  bind: typeof bind,
};


export const Update = {
  ap,
  of,
  patch,
  modify,
  effect,
  get,
  batch,
  concat,
  Component,
  bind,
} as UpdateStatic;

export interface BoundStatics<State> {
  ap<A,B,C>(a: Update<State,A>, b: Update<State,B>, f: (a: A, b: B) => C): Apply<State,C>;
  ap<A,B,C,D>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, f: (a: A, b: B, c: C) => D): Apply<State,D>;
  ap<A,B,C,D,E>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, f: (a: A, b: B, c: C, d: D) => E): Apply<State,E>;
  ap<A,B,C,D,E,F>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, e: Update<State,E>, f: (a: A, b: B, c: C, d: D, e: E) => F): Apply<State,F>;
  ap<A,B,C,D,E,F,G>(a: Update<State,A>, b: Update<State,B>, c: Update<State,C>, d: Update<State,D>, e: Update<State,E>, f_: Update<State,F>, f: (a: A, b: B, c: C, d: D, e: E, f: F) => G): Apply<State,G>;

  batch(...steps: Update<State, any>[]): Batch<State, void>;
  batch(steps: Update<State, any>[]): Batch<State, void>;  

  concat(...steps: Update<State, any>[]): Concat<State, void>;
  concat(steps: Update<State, any>[]): Concat<State, void>;

  of<A>(value: A): Pure<State,A>;
  patch(patch: Partial<State>, callback?: () => void): Patch<State, void>;
  modify(proj: (x: State) => State, callback?: () => void): Modify<State, void>;
  effect<Error, Success>(eff: Eff<Error, Success>): Effect<State, Either<Error, Success>>;  
  get: Get<State, State>;
}

export function bind<State>() {
  return Update as any as BoundStatics<State>;
}
