import * as t from '../decoder';
import { camelCase } from 'lodash';
import { absurd } from '../types';


/** 
 * Источник данных для чтения конфигов
 */
export type ConfigSource =
  | { tag: 'Cli', args: string[], prefix: string }
  | { tag: 'Location/search', value: string, transformKey(x: string): string }
  | { tag: 'JSON', value: object }
  | { tag: 'Concat', sources: ConfigSource[] }


export function cli(args: string[], prefix: string = ''): ConfigSource {
  return { tag: 'Cli', args, prefix };
}


export function queryString(value: string, transformKey = camelCase): ConfigSource {
  return { tag: 'Location/search', value, transformKey };
}


export function json(value: object): ConfigSource {
  return { tag: 'JSON', value };
}


export function concat(...sources: ConfigSource[]): ConfigSource {
  return { tag: 'Concat', sources };
}


export function validate<A>(config: ConfigSource, decoder: t.RecordDecoder<A>) {
  return decoder.validate(merge(decoder, config));
}


export function merge<A>(decoder: t.RecordDecoder<A>, ...srcs: ConfigSource[]): object {
  const source = srcs.length === 1 ? srcs[0] : concat(...srcs);
  if (source.tag === 'Cli') {
    const value: Record<string, any> = {};
    for (let i = 0; i < source.args.length; i++) {
      const prefixLen = '--'.length + source.prefix.length;
      const rest = source.args[i].substr(prefixLen);
      const [k, v] = rest.indexOf('=') === -1 ? [rest, source.args[++i] || ''] : source.args[i].split('=');
      // Проверка того что это известно
      if (!decoder.description.hasOwnProperty(k)) continue;
      value[k] = decoderFromString(decoder.description[k], v);
    }
    return value;    
  }

  if (source.tag === 'Location/search') {
    const substring = source.value[0] === '?' ? source.value.substr(1) : source.value;
    const value = substring.split('&').map(x => x.split('=')).reduce((acc, [k, v]: any) => {
      const key = source.transformKey(decodeURIComponent(k));
      if (!decoder.description.hasOwnProperty(key)) return acc;
      acc[key] = decoderFromString(decoder.description[key], decodeURIComponent(v));
      return acc;
    }, {});
    return value;
  }

  if (source.tag === 'JSON') {
    return source.value;
  }

  if (source.tag === 'Concat' ) {
    // @ts-ignore
    return Object.assign({}, ...source.sources.map(x => merge(decoder, x)));
  }
  
  return absurd(source);
}


export function decoderFromString(decoder: t.Decoder<any>, value: string): any {
  if (decoder instanceof t.CustomDecoder) {
    throw new Error('[config-reader] CustomDecoder is not supported');
  }

  if (decoder instanceof t.ArrayDecoder) {
    return value.split(', ').map(x => decoderFromString(decoder.decoder, x));
  }

  if (decoder instanceof t.DictionaryDecoder) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }
  
  if (decoder instanceof t.RecordDecoder) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }
  
  if (decoder instanceof t.AtDecoder) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }

  if (decoder instanceof t.PrimitiveDecoder) {
    if (decoder.primitive === 'string') return value;
    if (decoder.primitive === 'boolean') return !(value === 'false' || value === '0');
    if (decoder.primitive === 'undefined') return undefined;
    if (decoder.primitive === 'null') return null;
    if (decoder.primitive === 'any') return value;
    if (decoder.primitive === 'nat') return Number(value);
    if (decoder.primitive === 'int') return Number(value);
    if (decoder.primitive === 'float') return Number(value);
    return absurd(decoder.primitive);
  }
  
  if (decoder instanceof t.PureDecoder) {
    return value;
  }
  
  if (decoder instanceof t.ChainDecoder) {
    throw new Error('[config-reader] ChainDecoder is not supported');
  }
  
  if (decoder instanceof t.OneOfDecoder) {
    throw new Error('[config-reader] OneOfDecoder is not supported');
  }
  
  if (decoder instanceof t.DiscriminatedDecoder) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }

  if (decoder instanceof t.HasDecoder) {
    if (decoder instanceof t.WithDefaultDecoder) {
      return decoderFromString(decoder.decoder, value);
    }
    
    return decoderFromString(decoder.toDecoder(), value);
  }

  return absurd(decoder);
}
