import * as t from '../decode';
import { Either, success, failure } from '../either';
import * as readline from 'readline';


/// пример кастомного декодера
const loginDecoder = t.decoder<string>('login', value => {
  if (typeof(value) !== 'string') return failure('логин должен быть строкой');
  if (value.length < 3) return failure('длина логина должна быть не меньше 3 символов');
  if (value.length > 9) return failure('длина логина должна быть не больше 9 символов');
  if (/\d/.test(value[0])) return failure('логин не должен начинаться с цифры');
  if (!/[\w_\-]+/.test(value)) return failure('логин может содержать только буквы, цифры и знаки -_');
  return success(value);
});


/// можно использовать предопределенные декодеры для предварительной проверки
const passwordDecoder = t.string.chain(value => t.decoder(() => {
  if (value.length < 6) return failure('длина пароля должна быть не меньше 6 символов');
  if (!/\d+/.test(value)) return failure('пароль должен содержать тотябы одну цифру');
  if (!/\a+/.test(value)) return failure('пароль должен содержать тотябы одну букву');
  return success(value);
}));


/// простые декодеры можно комбинировать в более сложные с помошью `t.record`, `t.ap`, `t.oneOf` и тд
const formDecoder = t.record({
  login: loginDecoder,
  password: passwordDecoder,
});


const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});


function validateJSON(str: string): Either<string, any> {
  try {
    return success(JSON.parse(str));
  } catch {
    return failure('невалидный JSON');
  }
}

rl.setPrompt('Форма регистрации: введите логин и пароль в виде json строки {"login":"…", "password":"…"}\n> ');
rl.on('line', (answer) => {
  console.log(validateJSON(answer).chain(x => formDecoder.validate(x)));
  rl.prompt();
});
rl.prompt();
