import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import withStyles, { StyleRules } from '@material-ui/core/styles/withStyles';
import { FieldProps } from '~/create-form';
import * as moment from 'moment';
import TextField, { Props as TextFieldProps } from './TextField';
import { StandardProps, Popover } from '@material-ui/core';
import Icon from './Icon';
import 'react-dates/initialize';
import 'react-dates/lib/css/_datepicker.css';
import DayPickerSingleDateController from 'react-dates/lib/components/DayPickerSingleDateController';


// Props
export type Props = StandardProps<React.HTMLProps<HTMLDivElement>, ClassKey, 'value'|'disabled'> & FieldProps<moment.Moment> & {
  InputProps?: Partial<TextFieldProps>;
  format?: string;
  nonNull?: boolean;
  dayPickerProps?: Partial<Record<string, any>>;
}


// Model
export interface State {
  textFieldValue: string|null;
  open: boolean;
  value: null|string;
}


// Component
// @ts-ignore
@withStyles(styles)
class MomentField extends React.Component<Props, State> {
  static defaultProps = {
    nonNull: false,
  } as any;
  
  rootEl: HTMLElement|null;
  pickerEl: HTMLElement|null;
  state: State = { open: false, textFieldValue: null, value: null };
  unlisten: Function|null;

  componentDidMount() {
    document.addEventListener('focusin', this.handleFocusIn);
    document.addEventListener('click', this.handleFocusIn);
    this.unlisten = () => (document.removeEventListener('focusin', this.handleFocusIn), document.removeEventListener('click', this.handleFocusIn), this.unlisten = null);
  }

  componentWillUnmount() {
    this.unlisten && this.unlisten();
  }

  handleRootRef: React.Ref<HTMLDivElement> = c => {
    if (!c) { this.rootEl = null; return; }
    const rootEl = ReactDOM.findDOMNode(c); if (!rootEl) return;
    this.rootEl = rootEl as HTMLElement;
  };

  handlePickerRef: React.Ref<HTMLDivElement> = c => {
    if (!c) { this.pickerEl = null; return; }
    const rootEl = ReactDOM.findDOMNode(c); if (!rootEl) return;
    this.pickerEl = rootEl as HTMLElement;
  }

  handleChange: TextFieldProps['onValueChange'] = value => {
    this.setState({ value });
  };

  handleBlur = () => {
    const { onValueChange } = this.props;
    const { value } = this.state;
    if (!value) return;
    const nextValue = moment(value, this.format(this.props)); if (!nextValue.isValid()) return;
    this.setState({ value: null })
    const prevValueOf = this.props.value ? this.props.value.valueOf() : 0;
    const nextValueOf = nextValue.valueOf();
    prevValueOf !== nextValueOf && onValueChange && onValueChange(nextValue);
  };

  handleKeyDown: TextFieldProps['onKeyDown'] = e => {
    switch (e.key) {
      case 'Enter':
        this.handleBlur();
        break;
    }
  };

  handleFocusIn = (e: FocusEvent) => {
    const { rootEl, pickerEl } = this;
    const { disabled } = this.props; if (disabled) return;
    const targetElement = e.target as Element;
    if (targetElement.hasAttribute('data-ignore-focus-in')) return;
    const open = isElementInside(targetElement);
    this.setState({ open });

    function isElementInside(node: Node): boolean {
      if (node === rootEl || node === pickerEl) return true;
      return node.parentNode ? isElementInside(node.parentNode) : false;
    }
  };

  handleClear = (e) => {
    e.preventDefault(); e.stopPropagation();
    const { onValueChange } = this.props;
    onValueChange && onValueChange(null);
  };

  handleVisibilityToggle = (e: React.MouseEvent<HTMLSpanElement>) => {
    const { open } = this.state;
    this.setState({ open: !open });
  };

  handleClose = (e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (this.state.open) this.setState({ open: false });
  };

  handleDatesChange = (date: moment.Moment) => {
    const { value, onValueChange } = this.props;
    const prev = value || moment().startOf('minute');
    const next = date.clone();
    next.hours(prev.hours());
    next.minutes(prev.minutes());
    onValueChange && onValueChange(next);
  };

  format(props: Props): string {
    return props.format || 'DD/MM/YYYY';
  }

  render() {
    const { classes, ctx, error, InputProps, onBlur, onChange, onFocus, value, format: formapProps, disabled, nonNull, dayPickerProps, ...rest } = this.props;
    const { rootEl } = this;
    const { open } = this.state;
    const self = this;

    const endAdornment = <React.Fragment>
      <Icon data-ignore-focus-in onClick={this.handleVisibilityToggle}>date_range</Icon>
      {!nonNull && <Icon data-ignore-focus-in onClick={this.handleClear}>close</Icon>}
    </React.Fragment>;

    return <div {...rest} ref={this.handleRootRef}>
      <TextField
        {...InputProps as any}
        disabled={disabled}
        error={error}
        className={classes!.input}
        onValueChange={this.handleChange}
        onBlur={this.handleBlur}
        onKeyDown={this.handleKeyDown}
        endAdornment={endAdornment}
        value={inputValue(this.state, this.props)}
      />
      {rootEl && <Popover anchorEl={rootEl} open={open} onClose={this.handleClose} anchorOrigin={{ vertical: 'bottom', horizontal: 'left' }} disableAutoFocus disableEnforceFocus disableRestoreFocus hideBackdrop>
        <DayPickerSingleDateController
          ref={this.handlePickerRef}
          date={value}
          onDateChange={this.handleDatesChange}
          numberOfMonths={1}
          {...dayPickerProps}
        />
      </Popover>}
    </div>;
    
    function inputValue(state: State, input: FieldProps) {
      return state.value !== null ? state.value : input.value ? input.value.format(self.format(self.props)) : '';
    }
  }
}

export default MomentField;


// CSS классы
export type ClassKey = 'input';


// Styles
export function styles(theme: Theme): StyleRules<ClassKey> {
  // const { unit } = theme.spacing;

  return {
    input: {
      width: 180,
      position: 'relative',
    },
  };
}
