import * as Rx from 'rxjs';
import { defaultRoute, parser } from '~/config';
import * as eff from '../../eff';
import { IO } from '../../types';

export type RouteOut = typeof parser._O;
export type RouteIn = typeof parser._I;

/**
 * Возвращает хлебные крошки в виде `RouteOut[]`
 * @param location
 */
export function parseLocation(location: Location): RouteOut[] {
  if (!location.hash) { return [defaultRoute]; }
  const breadcrumbs = parser.parseAll(location.hash.substr(1));
  return breadcrumbs.length === 0 ? [defaultRoute] : breadcrumbs;
}
/**
 * Печатает `RouteOut` в строку
 * @param route роут
 */
export function print(route: RouteIn): string {
  return `#${parser.print(route)}`;
}

/**
 * Подписка на изменения `#` роута
 */
export const routeStream = new Rx.Subject<RouteOut[]>();

window.onpopstate = (() => { routeStream.next(parseLocation(window.location)); });

export function routePush(route: RouteIn, title: string = ''): IO<void> {
  window.history.pushState(null, title, print(route));
  routeStream.next(parseLocation(window.location));
}

export const routePush$ = (route: RouteIn, title: string = '') => eff.forget(routePush, route, title);

export function routeReplace(route: RouteIn, title: string = ''): IO<void> {
  window.history.replaceState(null, title, print(route));
  routeStream.next(parseLocation(window.location));
}

export const routeReplace$ = (route: RouteIn, title: string = '') => eff.forget(routePush, route, title);
