import * as eff from '~/common/eff';
import { Ctx as TranslationCtx, Translations } from '~/common/gettext';
import { Cache as LocalStorage } from '~/common/persistent';
import * as bcp47 from '~/common/utils/bcp47';
import { defaultLocale, localeCacheKey, localeDecoder, localePath, locales } from '~/config';
import { Async, LocalStorageError, TranslationLoadError } from '../error';

export type Locale = typeof localeDecoder['_A'];
export type Locales = Locale[];

export const cache = new LocalStorage<Locale>(localeCacheKey, localeDecoder);
/**
 * Возвращает в случаи успеха `Translations`. Необходим для работы с `gettext`
 * @param locale вариант перевода из `Locales`
 */
export function loadTranslations(locale: Locale): Async<Translations> {
  const translations: Translations | undefined = require(`${localePath('~')}${locale}.po`);
  return translations ? eff.success(translations) : eff.failure(TranslationLoadError(locale));
}
/**
 * Возвращает подходящую локаль из `window.navigator.languages` или `defaultLocale`
 */
export function BrowserLocale(): Locale {
  return (window.navigator.languages ? window.navigator.languages.slice() : [])
    .map((locale) => bcp47.lookupLocale(locale, locales)) // Приводит к BCP47, если это возможно
    .filter(browser => locales.find(locale => locale === browser))[0]
  || defaultLocale;
}
/**
 * Контекст для Locale
 */
export type Ctx = TranslationCtx & { locale: Locale };
/**
 * Инициализация локализации приложения по следущей логике:
 * Если в `LocalStorage` присутсвует установленная локаль, подгружает ее.
 * В случаи если `LocalStorage` значение отсуствует, выбор предоставляется из `локали браузера`
 * @param locale вариант перевода из `Locales`
 */
export function init(): Async<Ctx> {
  return cache.read().onError(e => {
    switch (e.tag) {
      case 'NoItem': return eff.success(BrowserLocale());
      default: return eff.failure(LocalStorageError(e));
    }
  }).chain((locale) => loadTranslations(locale)
    .map(translations => ({ translations, locale })),
  );
}
