import * as React from 'react';
import { Omit } from '@material-ui/core';
import memoize from '~/functions/memoize';


// Делимитер контекста
const CONTEXT_DELIMITER = String.fromCharCode(4);


// Translations
export type Translations = Record<string, Record<string, string[]|string>>;
export type GettextData = {
  '': Record<string, unknown>;
  '%podata': GettextExplicitKey[],
} & {
  [K in string]: string|string[];
};


// Содержимое поля `%podata`
export type GettextExplicitKey = {
  msgid: string;
  msgid_plural?: string;
  msgctxt?: string;
  msgstr: string|string[];
}


// Хелпер для перевода
export type Gettext = (singular_key: string, context?: string, plural_key?: string, val?: number) => string;
export type DeferredGettext = (singular_key: string, context?: string, plural_key?: string, val?: number) => I18nString;
export type I18nString = (locale?: string|LocaleCtx|null) => string;


// Использование локали из `ctx`
export type LocaleCtx = { locale: string };


/**
 * Создание хелпера для переводов
 */
export function makeGettext(...webpackContexts): DeferredGettext {
  const translations: Translations = assignData({}, ...flattenModules(webpackContexts));

  return (singular_key: string, context?: string, plural_key: string = singular_key, n: number = 1) => locale_ => {
    let locale = !locale_ ? 'en-US' : typeof(locale_) === 'string' ? locale_ : locale_.locale;
    locale = locale.replace(/_/, '-');
    if (!(locale in translations)) {
      // Не найдена локаль в `translations`
      return n === 1 ? singular_key : plural_key;
    }
    const key  = context ? context + CONTEXT_DELIMITER + singular_key : singular_key;
    const dict = translations[locale];
    const pluralIdx = getPluralForm(locale)(n);
    const pluralList = dict[key] || [singular_key, plural_key];
    return nth(pluralList, pluralIdx);
  };

  function flattenModules(xs: any[]) {
    return Array.prototype.concat.apply([], xs.map(x => {
      if ('__esModule' in x) x = x.default;
      return Array.isArray(x) ? flattenModules(x) : [x]
    }));
  }

  function nth<A>(xs: A|A[], idx: number): A {
    if (!Array.isArray(xs)) return xs;
    return idx >= xs.length ? xs[xs.length - 1] : xs[idx];
  }
}


/**
 * HOC для добавления переводов (для разделения переводов по фичам)
 * ```ts
 * export default withGettext(require('./i18n'))();
 * ```
 */
export function withGettext(...webpackContexts) {
  const gettext = typeof(webpackContexts[0]) === 'function' ? webpackContexts[0] : makeGettext(...webpackContexts);
  return <P extends { __: Gettext, ctx?: { locale: string } }>(Component: React.ComponentType<P>) => {
    class WithTranslations extends React.Component<Omit<P, '__'>> {
      makeGettext: (locale?: string|LocaleCtx) => Gettext = memoize((locale) => (...args) => gettext.apply(void 0, args)(locale))
      
      render() {
        const { ctx } = this.props;
        // @ts-ignore
        const locale = ctx ? (ctx.locale || 'en-US') : 'en-US';
        return React.createElement(Component as any, { ...this.props as any, __: this.makeGettext(locale) });
      }
    }
    // @ts-ignore
    WithTranslations.displayName = 'WithTranslations(' + Component.name + ')';
    require('hoist-non-react-statics')(WithTranslations, Component);
    return WithTranslations;
  };
}


/**
 * Хелпер для импорта нескольких переводов
 * ```ts
 * export default requireTranslations(require.context('./i18', false, /\.po$/));
 * ```
 */
export function requireTranslations(webpackContext /*: WebpackContext*/): Translations {
  return webpackContext.keys().reduce((acc, k) => mergeTranslations(acc, webpackContext(k)), {});

  function mergeTranslations(acc, data) {
    data = data.__esModule ? data.default : data;
    for (const k of Object.keys(data)) {
      acc[k] = acc[k] || {};
      Object.assign(acc[k], data[k]);
    }
    return acc;
  }
}


/**
 * Добавление переводов из `srcs` в `dst`
 */
function assignData(dst: Translations, ...srcs: GettextData[]): Translations {
  srcs.forEach(data => {
    // @ts-ignore
    if ('locale_data' in data) data = data['locale_data']['messages'];
    const locale_ = data[''].language || data[''].lang || data[''].locale; if (!locale_ || typeof(locale_) !== 'string') return;
    const locale = locale_.replace(/_/g, '-');
    const podata = data['%podata'];
    for (const k of Object.keys(data)) {
      if (k === '' || k === '%podata') continue;
      dst[locale] = dst[locale] || {};
      if (Array.isArray(data[k]) && data[k][0] === null) {
        dst[locale][k] = data[k].slice(1);
        if (dst[locale][k].length === 1) dst[locale][k] = dst[locale][k][0];
      } else {
        dst[locale][k] = data[k];
      }
    }

    if (podata) for (const explicit of podata) {
      const key = explicit.msgctxt ? explicit.msgctxt + CONTEXT_DELIMITER + explicit.msgid : explicit.msgid;
      dst[locale] = dst[locale] || {};
      dst[locale][key] = explicit.msgstr;
    }
  });
  return dst;  
}


/**
 * Минималистичный `sprintf`. Только простая замена %s на позиционные аргументы
 */
export function sprintf(format: string, ...args: any[]): string {
  var i = 0;
  return format.replace(/%(s|d)/g, function() {
    return args[i++];
  });
}


export namespace deferred {
  /**
   * Минималистичный `sprintf`. Только простая замена %s на позиционные аргументы
   */
  export function sprintf(format: I18nString, ...args: any[]): I18nString {
    return locale => {
      var i = 0;
      return format(locale).replace(/%(s|d)/g, function() {
        return args[i++];
      });
    }
  }
}


/**
 * Вычисление множественной формы для указанного языка. Если язык не
 * распознан применяются правила для
 * https://developer.mozilla.org/en-US/docs/Mozilla/Localization/Localization_and_Plurals#Plural_rule_1_(2_forms)
 * http://www.unicode.org/cldr/charts/latest/supplemental/language_plural_rules.html
 */
export function getPluralForm(locale: string): (n: number) => number {
  const [lang] = locale.split(/[_\-]/);

  // Славянские языки, 3 формы
  if (lang === 'ru' || lang === 'be' || lang === 'bs' || lang === 'hr' || lang === 'sh' || lang === 'sr_ME' || lang === 'uk') {
    return n => (n % 10 == 1 && n % 100 != 11) ? 0 : ((n % 10 >= 2 && n % 10 <= 4 && (n % 100 < 12 || n % 100 > 14)) ? 1 : 2);
  }

  // Литовский
  if (lang === 'lt') {
    return n => (n % 10 == 1 && (n % 100 < 11 || n % 100 > 19)) ? 0 : ((n % 10 >= 2 && n % 10 <= 9 && (n % 100 < 11 || n % 100 > 19)) ? 1 : 2);
  }

  // Арабский 6 форм
  if (lang === 'ar' || lang === 'ars') {
    return n => (n == 0) ? 0 : ((n == 1) ? 1 : ((n == 2) ? 2 : ((n % 100 >= 3 && n % 100 <= 10) ? 3 : ((n % 100 >= 11 && n % 100 <= 99) ? 4 : 5))));
  }

  // Бретонский 5 форм
  if (lang === 'br') {
    return n => (n % 10 == 1 && n % 100 != 11 && n % 100 != 71 && n % 100 != 91) ? 0 : ((n % 10 == 2 && n % 100 != 12 && n % 100 != 72 && n % 100 != 92) ? 1 : ((((n % 10 == 3 || n % 10 == 4) || n % 10 == 9) && (n % 100 < 10 || n % 100 > 19) && (n % 100 < 70 || n % 100 > 79) && (n % 100 < 90 || n % 100 > 99)) ? 2 : ((n != 0 && n % 1000000 == 0) ? 3 : 4)));
  }

  // Чешский и словакский, 3 формы
  if (lang === 'cz' || lang === 'sk') {
    return n => (n == 1) ? 0 : ((n >= 2 && n <= 4) ? 1 : 2);
  }

  // Уельский, 6 форм
  if (lang === 'cy') {
    return n => (n == 0) ? 0 : ((n == 1) ? 1 : ((n == 2) ? 2 : ((n == 3) ? 3 : ((n == 6) ? 4 : 5))));
  }
  
  // Словенский, лужицкие сербы, 4 формы
  if (lang === 'sl' || lang === 'dsb' || lang === 'hsb') {
    return n => (n % 100 == 1) ? 0 : ((n % 100 == 2) ? 1 : ((n % 100 == 3 || n % 100 == 4) ? 2 : 3));
  }

  // TODO http://mlocati.github.io/cldr-to-gettext-plural-rules/
  return n => n == 1 ? 0 : 1;
}
