import * as eff from '../eff';
import { Eff } from '../eff';
import * as t from '../decode';
import * as http from '../http';
import { HttpError } from '../http';
import { Either } from '../either';
import Monad from 'burrido';


/// shim для nodejs
if (typeof(XMLHttpRequest) === 'undefined') global['XMLHttpRequest'] = require('xhr2');


// ---------------------------------------------------------------------------
// Пример do-нотации с еффектами.
// 
// Do-нотация позволяет развернуть монадные вычисления (цепочки с
// методом `chain`) в императивный код. Синтаксис acync/await —
// частный случай do-нотации для промисов.
//
// пример do-нотации:
// const process = Do(function*(){
//   const x = yield fooAction;
//   const y = yield barAction;
//   const z = yield bazAction;
//   return `x is ${x}, y is ${y} and z is ${z}`;
// });
// 
// аналогичный код с использованием `chain`
// const process = fooAction.chain(
//   x => barAction.chain(
//     y => bazAction.chain(
//       z => of(`x is ${x}, y is ${y} and z is ${z}`)
//     )
//   )
// );
// ---------------------------------------------------------------------------

const EffMonad = Monad({
  pure: eff.of,
  bind: (xs, f) => xs.chainEff(f),
});


/// https://api.random.org/json-rpc/1/basic
function randomOrgMethod<a>(method: string, params: object, decoder: t.Decoder<a>): Eff<HttpError, a> {
  return http.send({
    url: 'https://api.random.org/json-rpc/1/invoke',
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: { "jsonrpc": "2.0", "method": method, "params": { ...params, "apiKey": "8e5b7dd6-fe16-41f1-842d-57481c2777b0" }, "id": 42 }
  }).chainEff((x => (http.expectJSON(t.at(['result', 'random', 'data'], decoder))(x))));
}


const generateIntegers = (n: number, min: number, max: number) => randomOrgMethod('generateIntegers', { n, min, max }, t.array(t.nat));
const generateStrings = (n: number, length: number, characters: string) => randomOrgMethod('generateStrings', { n, length, characters }, t.array(t.string));
const generateUUIDs = (n: number) => randomOrgMethod('generateUUIDs', { n }, t.array(t.string));
const generateElements = <a>(array: a[]) => generateIntegers(1, 0, array.length - 1).mapEff(([idx]) => array[idx]);


/// тип результата
type Result =
  | { tag: 'Integers', data: number[] }
  | { tag: 'Strings', data: string[] }
  | { tag: 'UUIDs', data: string[] }


/// пример do-нотации, выведение типов не работает
const randomOrgEff: Eff<HttpError, Result> = EffMonad.Do(function*(){
  const tag = yield generateElements(['Integers', 'Strings', 'UUIDs']);
  let data = undefined;
  switch (tag) {
    case 'Integers': data = yield generateIntegers(10, 0, 10000); break;
    case 'Strings': data = yield generateStrings(4, 10, 'abcdefjhijklmnopqrstuvwxyz'); break;
    case 'UUIDs': data = yield generateUUIDs(3); break;
  }
  return { tag, data };
});


function resultToString(ethr: Either<HttpError, Result>): string {
  if (ethr.tag === 'Left') return 'Error: ' + ethr.value.tag
  switch (ethr.value.tag) {
    case 'Integers': return 'Integers: ' + ethr.value.data.join(',');
    case 'Strings': return 'Strings: ' + ethr.value.data.join(',');
    case 'UUIDs': return 'UUIDs: ' + ethr.value.data.join(',');
  }
}


/// еффект, как любой Observable можно запустить несколько раз
for (let i = 0; i < 10; i++) randomOrgEff.subscribe(ethr => console.log(`randomOrgEff #${i}:`, resultToString(ethr)));

