import * as React from 'react';
import { CrudResource } from '~/jsonapi';
import { AuthCtx as Ctx } from '~/utils';
import { Eff } from '@bitmaster/core';
import * as Rx from 'rxjs';
import { Err, notifyError } from '~/context';
import { FieldProps } from '~/fields';
import * as F from '~/fields';
import { Resource } from '@bitmaster/utils/jsonapi';


// Props
export type Props<A = any> = {
  children: React.ReactElement<{ resource: CrudResource<A, never> }>;
  source: PrimarySource<A>;
  id?: string;
}


// Props
export type State<A extends Resource<any>= any> = {
  value: A|null; // Ресурс полученный из id
  pending: boolean;
}


// Источник для получения ресурса по ид
export type PrimarySource<A> = {
  getPrimary(ctx: Ctx, id: string): Eff<Err, A>;
}


// Component
export class ResourceID extends React.Component<Props, State> {
  subscription: Rx.Subscription|null; // Подписка на запрос
  state: State = { pending: false, value: null };
  inputContext: FieldProps|null = null;
  prevValue: any;

  projectContext = (input: FieldProps) => {
    this.inputContext = input;
    const { value } = this.state;
    if (this.prevValue !== input.value) {
      if (input.value) this.resolveResource(input);
      else this.setState({ value: null });
      this.prevValue = input.value;
    }
    return { ...input, value, onValueChange: this.handleValueChange };
  };

  handleValueChange = (value: State['value'], at=[]) => {
    const id = this.props.id || 'id';
    const nextValue = value ? value[id] : null;
    this.setState({ value, pending: false });
    if (!this.inputContext) return;
    const { onValueChange } = this.inputContext;
    if (this.subscription) {
      this.subscription.unsubscribe();
      this.subscription = null;
    }
    onValueChange && onValueChange(nextValue, []);
  };

  componentDidMount() {
    this.inputContext && this.resolveResource(this.inputContext);
  }
  
  resolveResource(inputContext: FieldProps) {
    if (this.subscription) return;
    const { source } = this.props; if (!source) return;
    const { ctx, value } = inputContext; if (!ctx) return; if (typeof(value) !== 'string') return;
    this.setState({ pending: true });
    this.subscription = source.getPrimary(ctx, value).subscribe(ethr => {
      this.subscription = null;
      if (ethr.tag === 'Left') {
        this.setState({ pending: false });
        notifyError(ethr.value);
      } else {
        this.setState({ pending: false, value: ethr.value });
      }
    });
  }
  
  render() {
    return <F.Modifier proj={this.projectContext}>
      {this.props.children}
    </F.Modifier>;
  }
}


export default ResourceID;
