import { Eff, eff, Either, Decoder, success, failure, decode as t } from '../core';
import * as aesjs from 'aes-js';


/** options for `Cache` */
export interface CacheOptions {
  key?: number[] | Uint8Array;
  version?: string;
  lifetime?: number;
  expires?: number;
}


/** possible errors */
export type Err =
  | { tag: 'NoItem' }
  | { tag: 'SizeLimitExeeded' }
  | { tag: 'InvalidPayload', problem: t.Problem }
  | { tag: 'VersionMismatch', actual: string|null, expected: string }


/** persistent cache */
export class Cache<A> {
  readonly _A: A;

  constructor(
    public name: string, // should be unique
    public decoder: Decoder<A>,
    public options: CacheOptions = {},
  ) {}

  saveSync(payload: A): Either<Err,null> {
    try {
      const updatedAt = Date.now();
      const createdAt = Date.now();
      const version = this.options.version || null;
      let stringContent = JSON.stringify({ updatedAt, createdAt, version, payload });
      if (this.options.key) {
        const enc = new aesjs.ModeOfOperation.ctr(this.options.key);
        const bytes = aesjs.utils.utf8.toBytes(stringContent);
        stringContent = aesjs.utils.hex.fromBytes(enc.encrypt(bytes));
      }

      localStorage.setItem(this.name, stringContent);
    } catch(e) {
      return failure({ tag: 'SizeLimitExeeded' } as Err);
    }
    return success(null);
  }

  save(payload: A): Eff<Err,null> {
    return eff.lazy(() => this.saveSync(payload));
  }

  readSync(): Either<Err,A> {
    let stringContent = localStorage.getItem(this.name);
    if (this.options.key && stringContent !== null) {
      const enc = new aesjs.ModeOfOperation.ctr(this.options.key);
      const bytes = aesjs.utils.hex.toBytes(stringContent)
      stringContent = aesjs.utils.utf8.fromBytes(enc.decrypt(bytes));
    }

    if (stringContent === null) return failure({ tag: 'NoItem' } as Err);
    try {
      const json = JSON.parse(stringContent);
      // @ts-ignore
      return cacheItemDecoder<A>(this.decoder).validate(json)
        .mapLeft(problem => ({ tag: 'InvalidPayload', problem } as Err))
        .chain(({payload, version}) => {
	  if (this.options.version && version !== version) {
	    return failure<Err>({ tag: 'VersionMismatch', expected: this.options.version, actual: version })
	  }
	  return success(payload);
	});
    } catch (e) {
      return failure({ tag: 'InvalidPayload', problem: e.message } as Err);
    }
  }

  read(): Eff<Err,A> {
    return eff.lazy(() => this.readSync());
  }

  clearSync(): void {
    localStorage.removeItem(this.name);
  }

  clear(): Eff<never, null> {
    return eff.lazy(() => (this.clearSync(), success(null)));
  }

  modifySync(f: (a: A) => A): Either<Err, null> {
    const orignal = this.readSync();
    if (orignal.tag === 'Left') return orignal as any;
    return this.saveSync(f(orignal.value));
  }

  modify(f: (a: A) => A): Eff<Err, null> {
    return eff.lazy(() => this.modifySync(f));
  }
}


// Data stored in storage
export interface CacheItem<A> {
  createdAt: number;
  updatedAt: number;
  version: string|null;
  payload: A;
}


// Decode a `CacheItem`
const cacheItemDecoder = <A>(decoder: Decoder<A>) => t.record({
  createdAt: t.float,
  updatedAt: t.float,
  version: t.oneOf(t.string, t.null),
  payload: decoder,
});
