import { isEqual, isEmpty } from 'lodash';
import { decode as t, either, success, failure } from '../core';
import { Expr } from '../types';
import * as decoders from './decoders';
import { Decoder, Validation } from '../decoder';
import { Left, Right } from '../either';
import { Params } from '../core/http';


/** Ресурс jsonapi */
export interface Resource<T extends string = string> {
  id: string;
  type: T;
}


/** Вспомогательный тип для построения GET параметров */
export interface Flags {
  fields: Record<string, StringSet>;
  include: StringSet;
}

export type StringSet = Record<string, true>;


/** DSL для jsonapi ресурсов */
export type JsonApi<A> =
  | Attr<A>
  | Related<A>
  | RelatedMany<A>
  | WithName<A>
  | WithDefault<A>
  ;


// Базовый класс для наследования методов
export class JsonApiBase<A> {
  // @ts-ignore
  readonly _A: A;

  /**
   * Установка дефолтного значения в случае отсутствия поля
   */
  withDefault(this: JsonApi<A>, def: A): WithDefault<A>;
  withDefault<B extends Expr>(this: JsonApi<A>, def: B): WithDefault<A|B>;
  withDefault<B extends Expr>(this: JsonApi<A>, def: B): WithDefault<A|B> {
    return new WithDefault(def, this);
  }

  /**
   * Создать `to-many` поле
   */
  many(this: Related<A>): RelatedMany<A[]>{
    return new RelatedMany(this as any);
  }

  /**
   * Сбор информации по полям ресурсов для построения GET запроса
   */
  collectFlags(this: Related<A>): Flags {
    const output: Flags = { fields: {}, include: {} };
    for (let key in this.desc) {
      if (!this.desc.hasOwnProperty(key)) continue;
      go(output, [], this.type, key, this.desc[key]);
    }
    return output;

    function go(acc: Flags, path: string[], ty: string, key: string, field: JsonApi<any>) {
      switch(field.tag) {
        case 'Attr': {
	  acc.fields[ty] = acc.fields[ty] || {};
	  acc.fields[ty][key] = true;
	  return;
	}
        case 'Related': {
	  acc.fields[ty] = acc.fields[ty] || {};
	  acc.fields[ty][key] = true;
          if (!isEmpty(field.desc)) acc.include[path.concat(key).join('.')] = true;
          for (let k in field.desc) {
	    if (!field.desc.hasOwnProperty(k)) continue;
	    go(acc, path.concat(key), field.type, k, field.desc[k]);
	  }
	  return;
        }
        case 'RelatedMany': {
	  acc.fields[ty] = acc.fields[ty] || {};
	  acc.fields[ty][key] = true;
          if (!isEmpty(field.child.desc)) acc.include[path.concat(key).join('.')] = true;
          for (let k in field.child.desc) {
	    if (!field.child.desc.hasOwnProperty(k)) continue;
	    go(acc, path.concat(key), field.child.type, k, field.child.desc[k]);
	  }
	  return;
        }
        case 'WithDefault': {
	  go(acc, path, ty, key, field.child);
	  return;
        }
        case 'WithName': {
	  go(acc, path, ty, field.name, field.child);
	  return;
        }
      }
    }
  }

  /**
   *  Построение GET запроса
   */
  collectQuery(this: Related<A>): Params {
    const { fields, include } = this.collectFlags();
    const output = { include: Object.keys(include).join(',') || undefined, } as Params;
    for (let k in fields) {
      output[`fields[${k}]`] = Object.keys(fields[k]).join(',') || undefined;
    }
    return output;
  }

  /**
   * Валидация jsonapi документа
   */
  validate(this: Related<A>, doc: decoders.Document, res: decoders.Resource): Validation<A> {
    if (res.type !== this.type) return failure(`invalid resource type: ${res.type}, expected ${this.type}`);
    const output: Record<string, any> = { id: res.id, type: res.type };
    for (let key in this.desc) {
      if (!this.desc.hasOwnProperty(key)) continue;
      const result = helperRec(key, this.desc[key], doc, res);
      if (result instanceof Left) return result;
      output[key] = result['value'];
    }
    return success(output as A);

    function helperRec(key: string, field: JsonApi<any>, doc: decoders.Document, res: decoders.Resource): Validation<any> {
      switch(field.tag) {
        case 'Attr': {
	  const attrs = res.attributes;
	  if (attrs === null) return failure(`resource ${res.type}#${res.id} doesn't have any attributes`);
	  return field.decoder.validate(attrs[key]);
        }
        case 'Related': {
	  if (isEmpty(field.desc)) return decoders.relatedLinkage(res, key);
	  return decoders.related(doc, res, key).chain(relatedRes => field.validate(doc, relatedRes));
        }
        case 'RelatedMany': {
	  return decoders.relatedCollection(doc, res, key).chain(rs => either.traverse(rs, x => field.child.validate(doc, x)));
        }
        case 'WithDefault': {
          const nested = field.child;
          switch(nested.tag) {
   	    case 'Related': {
	      if (isEmpty(nested.desc)) return decoders.relatedLinkage(res, key).fold(() => success(field.defaultValue), success);
              return decoders.related(doc, res, key).fold(() => success(field.defaultValue), relatedRes => nested.validate(doc, relatedRes));
            }
 	    case 'RelatedMany': return decoders.relatedCollection(doc, res, key).fold(() => success(field.defaultValue), rs => either.traverse(rs, x => nested.child.validate(doc, x)));
          }
	  const result = helperRec(key, field.child, doc, res);
	  if (result instanceof Right) return result;
	  return success(field.defaultValue);
        }
        case 'WithName': {
	  return helperRec(field.name, field.child, doc, res)
        }
      }
    }
  }

  /**
   *  Построение содержимого POST запроса
   */
  post(this: Related<A>, resource: A): object {
    const data: Record<string, any> = { type: this.type };
    const attributes: Record<string, any> = {};
    const relationships: Record<string, any> = {};
    for (let key in this.desc) {
      if (!this.desc.hasOwnProperty(key)) continue;
      postRec(key, key, this.desc[key]);
    }
    if (!isEmpty(attributes)) data.attributes = attributes;
    if (!isEmpty(relationships)) data.relationships = relationships;
    return { data };

    function postRec(ikey: string, okey: string, field: JsonApi<any>) {
      switch(field.tag) {
        case 'Attr':
	  attributes[okey] = resource[ikey];
	  return;
        case 'Related':
	  relationships[okey] = { data: { type: resource[ikey].type, id: resource[ikey].id } };
	  return;
        case 'RelatedMany':
	  relationships[okey] = { data: resource[ikey].map(x => ({ type: x.type, id: x.id })) };
  	  return;
        case 'WithDefault':
	  if (!isEqual(resource[ikey], field.defaultValue)) postRec(ikey, okey, field.child);
	  return;
        case 'WithName':
	  postRec(ikey, field.name, field.child);
	  return;
      }
    }
  }

  /**
   * Построение содержимого PATCH запроса
   */
  patch(this: Related<A>, id: string, patch: Partial<A>): object {
    const data: Record<string, any> = { type: this.type, id };
    const attributes: Record<string, any> = {};
    const relationships: Record<string, any> = {};
    for (let key in patch) {
      if (!this.desc.hasOwnProperty(key)) continue;
      patchRec(key, key, this.desc[key]);
    }
    if (!isEmpty(attributes)) data.attributes = attributes;
    if (!isEmpty(relationships)) data.relationships = relationships;
    return { data };

    function patchRec(ikey: string, okey: string, field: JsonApi<any>) {
      switch(field.tag) {
	case 'Attr':
	  attributes[okey] = patch[ikey];
	  return;
	case 'Related':
	  relationships[okey] = { data: { type: patch[ikey].type, id: patch[ikey].id } };
	  return;
	case 'RelatedMany':
	  relationships[okey] = { data: patch[ikey].map(x => ({ type: x.type, id: x.id })) };
	  return;
	case 'WithDefault':
	  if (isEqual(patch[ikey], field.defaultValue)) {
	    switch (field.child.tag) {
	      case 'Attr': attributes[okey] = field.defaultValue; return;
	      case 'Related': relationships[okey] = { data: null }; return;
	      case 'RelatedMany': relationships[okey] = { data: [] }; return;
	      default: console.warn(`invalid tag nested inside WithDefault ${field.child.tag}`); return;
	    }
	  }
	  patchRec(ikey, okey, field.child);
	  return;
	case 'WithName':
	  patchRec(ikey, field.name, field.child);
	  return;
      }
    }
  }

  /**
   * Декодер для primary документа
   */
  primaryDecoder(this: Related<A>): Decoder<A> {
    return t.decoder('primary', val => decoders.document.validate(val).chain(
      doc => decoders.primary(doc).chain(res => this.validate(doc, res))
    ));
  }

  /**
   * Декодер для коллекции документов
   */
  collectionDecoder(this: Related<A>): Decoder<A[]> {
    return t.decoder('collection', val => decoders.document.validate(val).chain(
      doc => ensureCollection(doc).chain(
        rs => either.traverse(rs, res => this.validate(doc, res))
      ))
    );

    function ensureCollection(doc: decoders.Document): Validation<Array<decoders.Resource>>{
      return Array.isArray(doc.data) ? success(doc.data) : failure(`jsonapi document doesn't contain collection of resorces`);
    }
  }

  /**
   * Расширение набора полей
   */
  extend<R extends ResourceRecord>(this: Related<A>, desc: R): Related<A & { [K in keyof R]: R[K]['_A'] }> {
    const replaceDecoderWithAttr = { ...this.desc } as Related<any>['desc'];
    if (desc) for (let k in desc) {
      if (!desc.hasOwnProperty(k)) continue;
      if (desc[k] instanceof t.DecoderBase) replaceDecoderWithAttr[k] = new Attr(desc[k] as any);
      else replaceDecoderWithAttr[k] = desc[k] as any;
    }
    return new Related(this.type, replaceDecoderWithAttr);
  }

  /** Переименование поля */
  withName(name: string): WithName<A> {
    return new WithName(name, this as any);
  }
}


/**
 * Атрибут ресурса
 */
export class Attr<A> extends JsonApiBase<A> {
  readonly tag: 'Attr' = 'Attr';

  constructor(
    readonly decoder: Decoder<A>,
  ) { super(); }
}


/**
 * Связанный ресурс
 */
export class Related<A> extends JsonApiBase<A> {
  readonly tag: 'Related' = 'Related';

  constructor(
    readonly type: string,
    readonly desc: Record<string, JsonApi<any>>,
  ) { super(); }
}


/**
 * Связанный ресурс (to-many)
 */
export class RelatedMany<A> extends JsonApiBase<A> {
  readonly tag: 'RelatedMany' = 'RelatedMany';

  constructor(
    readonly child: Related<any>, // A[number]
  ) { super(); }
}


/**
 * Поле с fallback значением
 */
export class WithDefault<A> extends JsonApiBase<A> {
  readonly tag: 'WithDefault' = 'WithDefault';

  constructor(
    readonly defaultValue: A,
    readonly child: JsonApi<any>,
  ) { super(); }
}


/**
 * Поле с переименованием
 */
export class WithName<A> extends JsonApiBase<A> {
  readonly tag: 'WithName' = 'WithName';

  constructor(
    readonly name: string,
    readonly child: JsonApi<any>,
  ) { super(); }
}


// Тип аргумента для `resource`
export type ResourceRecord = Record<string, Decoder<any>|JsonApi<any>>;


/**
 * Построение инстанса для `Related`
 */
export function resource<R extends ResourceRecord, T extends string>(type: T): Related<Resource<T>>;
export function resource<R extends ResourceRecord, T extends string>(type: T, desc: R): Related<Resource<T> & { [K in keyof R]: R[K]['_A'] }>;
export function resource<R extends ResourceRecord, T extends string>(type: T, desc?: R): Related<any> {
  const replaceDecoderWithAttr = {} as Related<any>['desc'];
  if (desc) for (let k in desc) {
    if (!desc.hasOwnProperty(k)) continue;
    if (desc[k] instanceof t.DecoderBase) replaceDecoderWithAttr[k] = new Attr(desc[k] as any);
    else replaceDecoderWithAttr[k] = desc[k] as any;
  }
  return new Related(type, replaceDecoderWithAttr);
}


/** Конструцтор атрибута */
export function attr<A>(decoder: Decoder<A>): Attr<A> {
  return new Attr(decoder);
}
