 * [Общее описание](#Общее-описание)
 * [Either](#Either)
 * [Decoder](#Decoder)
 * [Eff](#Eff)
 * [Контейнеры](#Контейнеры)
    - [Функтор (метод `map`)](#Функтор-(метод-`map`))
    - [Аппликативный функтор (фунция `ap`)](#Аппликативный-функтор-(фунция-`ap`))
    - [Монада (метод `chain`)](#Монада-(метод-`chain`))
    - [Traversable (метод `traverse`)](#Traversable-(метод-`traverse`))
    - [Alternative (метод `oneOf`)](#Alternative-(метод-`oneOf`))

--- 

# Общее описание

Библиотека является адаптацией [elm-архитектуры](https://guide.elm-lang.org/architecture/) для
языка [typescript](http://www.typescriptlang.org/), похожие библиотеки
— (https://github.com/gcanti/fp-ts), (https://github.com/gcanti/elm-ts)

```sh
$ tree examples src 
examples                 # примеры использования библиотеки
src
├── cmd.ts               # `Cmd` — контейнер для типа `Action`, используeтся в функции `update`
├── decode.ts            # библиотека для валидации JSON
├── eff.ts               # `Eff` — контейнер побочных эффектов
├── either.ts            # `Either` — контейнер для значений, которые могут содержать ошибку
├── http.ts              # обертка над XmlHttpRequest
├── index.ts             # импорт наиболее часто используемых функций
├── internal
│   └── expr.ts
├── optics.ts            # фунциональные линзы и призмы
├── option.ts            # `Option` или `Maybe` — контейнер для значений, которые могут отсутствовать
├── quickcheck           
│   ├── arbitrary.ts 
│   └── generator.ts
├── quickcheck.ts        # библиотека для генеративного тестирования
└── sub.ts

```

# Either

Сумма (или объединение) двух типов `err` и `succ`. `Either` нельзя определить как `type Either<err,succ> = err | succ;`
потомучто нельзя различить является ли тип `err` или `succ` когда
оба типа равны. `Either` часто используется как результат вычислений которые могут закончится
неудачно, `err` — тип результата с ошибкой, `succ` — тип успешного результата, 


```js
type Either<err,succ> = Left<err,succ> | Right<err,succ>;

// пример использования
function safeDiv(n: number, d: number): Either<string, number> {
        if (d === 0) return failure('Division by zero');
        return success(n / d);
}

const fiveByTwo = safeDiv(5, 2) // Right { value: 2.5 }
const oneByZero = safeDiv(1, 0) // Left { value: 'Division by zero' }
```

# Decoder

Позволяет декларативно описывать валидаторы для JSON. 


```js

type Validation<a> = Either<Problem, a>;
interface Decoder<a> {
        validate(value: any): Either<Problem, a>;
}

// пример использования
import * as t from './core/decode';

console.dir(t.string.validate(undefined)); // Left { value: 'Not a string' }
console.dir(t.string.validate('Hello, sailor!')); // Right { value: 'Hello, sailor!' }

const authDecoder = t.record({
        token: t.string,
        expires: t.date,
});

console.dir(authDecoder.validate(NaN)); // Left { value: 'not an object' }
console.dir(authDecoder.validate({ token: 'secret', date: '2017-08-23T15:08:31.244Z' })); // Right { value: { token: 'secret', date: '2017-08-23T15:08:31.244Z' } }

```

# Eff

Очень похож на `Promise`, содержит вычисления с побочными эффектами, вместо `then` используется `chain`


```js

export type Eff<err,succ> = Lazy<err,succ> | Async<err,succ>

// пример использования
import * as http from './core/http';


const effect = http.get('http://example.com/api/v1/login')
        .chainEff(() => get('http://example.com/api/v1/profile'))
        .chainEff(() => get('http://example.com/api/v1/settings));
        
effect.subscribe(); // выполнение еффекта

```

# Контейнеры 

В функциональном программировании вместо мутабельных классов и их инстансов активно используются
неизменяемые типы параметризованные другими типами «контейнеры типов». `Cmd`, `Eff`, `Either`,
`Decoder` являются примерами таких контейнеров. Для них определены несколько общих операций
(комбинаторов), наличие которых определяет принадлежность контейнера к определенному классу (не
путать с ООП классом), подобно тому как реализация определенных методов может означать реализацию
интерфейса в ООП-языках. Пример таких операций: `map`, `ap`, `chain`.


## Функтор (метод `map`)

Наиболее распространенный контейнер типов, функтор должен реализовать фунцию `map` имеющую тип
листинге ниже. Проще говоря функтор должен реализовать возможность применить чистую функцию к
значению хранящемся внутри него.

```js
function map<a,b>(a: F<a>, f: (a: a) => b): F<b>;

// пример использования в typesript
import { Either, success, failure } from './core/either';

const ethr: Either<string, number> = success(8);
const isEven = x => x % 2 === 0;

console.dir(ethr.map(isEven)); // Right { value: true }

```


## Аппликативный функтор (фунция `ap`)

Позволяет применить фунцию с произвольным количеством аргументов к такому же количеству контейнеров


```js
function of<a>(a: a): F<a>;
function ap<a,b,c>(a: F<a>, b: F<b>..., func: (a: a, b: b...) => c)): F<c>;

// пример использования в typescript
import { Either, success, failure } from './core/either';
import * as either from './core/either';

const one: Either<string, number> = success(1);
const two: Either<string, number> = success(2);
const three: Either<string, number> = success(3);
const falty: Either<string, number> = failure('Failed to get a value');

const sum1 = either.ap(one, two, three, (a, b, c) => a + b + c);
const sum2 = either.ap(one, two, falty, (a, b, c) => a + b + c);

console.dir(sum1); // Right { value: 6 }
console.dir(sum2); // Left { value: 'Failed to get a value' }

```


## Монада (метод `chain`)

Позволяет комбинировать вычисления подобно тому как промисы комбинируются через метод `then`


```js
function chain<a,b>(a: M<a>, func: (a: a) => M<b>)): M<b>;

// пример использования в typesript
import { Eff } from './core/eff';
import * as http from './core/http';

const body = { login: 'admin', password: 'admin' };
const authentificate = http.post('api/v1/login', { body })
        .chainEff(response => http.get('api/v1/profile'))
        .chainEff(response => http.get('api/v1/settings'));
        
authentificate.subscribe();        
        
```


## Traversable (метод `traverse`)

Некоторые контейнеры реализуют функцию `traverse` с помощью которой,
некоторые действия (определенные контейнером) можно применить к
массиву


```js
function traverse<a,b>(a: a[], func: (a: a) => T<b>)): T<b[]>;

// пример использования в typesript
import { Eff } from './core/eff';
import * as http from './core/http';


const resourceIds = ['10', '11', '12', '99'];
eff.traverse(resourceIds, id => http.get('api/v1/item', id));
        
authentificate.subscribe();        
        
```


## Alternative (метод `oneOf`)

Некоторые контейнеры реализуют функцию `oneOf` с помощью которой,
можно объединить несколько альтернативных вариантов


```js
function oneOf<a, b>(a: A<a>, b: A<b>): A<a|b>;

// пример использования в typesript
import { gen, Generator } from './core/quickcheck';

type Action = { tag: 'Increment' } | { tag: 'Decrement' } | { tag: 'Set', value: number }

const actionGenerator: Generator<Action> = gen.oneOf(
  gen.record({ tag: gen.of('Increment') }),
  gen.record({ tag: gen.of('Decrement') }),
  gen.record({ tag: gen.of('Set'), value: gen.nat }),
);
        
console.log(actionGenerator.ntimes(10).generate());
        
```
