import * as jed from 'jed';
import memoize from '../functions/memoize';


/** translations */
export interface Translations {
  locale_data: { messages: Record<string, any> };
  domain: string;
  debug?: boolean;
}


/** application context with translations */
export interface Ctx {
  translations: Translations|null;
}


/** For convenience also allow passing `Ctx` instead of just translations */
export type CtxOrMaybeTranslations = Ctx|Translations|null;


/** lazy string for deferred translation */
export type I18nString = (t: CtxOrMaybeTranslations) => string;


export type Gettext = (a: string) => string;


/** minimal sptrintf */
export function sprintf(format: string, ...args: any[]): string {
  var i = 0;
  return format.replace(/%(s|d)/g, function() {
    return args[i++];
  });
}


export function gettext(t: CtxOrMaybeTranslations, key: string): string {
  return getJedIntance(resolveTranslations(t)).gettext(key);
}


export function pgettext(t: CtxOrMaybeTranslations, context: string, key: string): string {
  return getJedIntance(resolveTranslations(t)).pgettext(context, key);
}


export function dcgettext(t: CtxOrMaybeTranslations, domain: string, key: string, context: string): string {
  return getJedIntance(resolveTranslations(t)).dcgettext(domain, key, context);
}


export function dcnpgettext(t: CtxOrMaybeTranslations, domain: string, context: string, single: string, plural: string, n: number): string {
  return getJedIntance(resolveTranslations(t)).dcnpgettext(domain, context, single, plural, n);
}


/** defered versions of functions */
export namespace deferred {

  export function gettext(key: string): I18nString {
    return t => getJedIntance(resolveTranslations(t)).gettext(key);
  }


  export function pgettext(context: string, key: string): I18nString {
    return t => getJedIntance(resolveTranslations(t)).pgettext(context, key);
  }


  export function dcgettext(domain: string, key: string, context: string): I18nString {
    return t => getJedIntance(resolveTranslations(t)).dcgettext(domain, key, context);
  }


  export function dcnpgettext(domain: string, context: string, single: I18nString, plural: I18nString, n: number): I18nString {
    return t => getJedIntance(resolveTranslations(t)).dcnpgettext(domain, context, single(t), plural(t), n);
  }

  export function sprintf(format: I18nString, ...args: any[]): I18nString {
    return translations => {
      const formatString = typeof(format) === 'string' ? format : format(translations);
      var i = 0;
      return formatString.replace(/%(s|d)/g, function() { return args[i++]; });
    }
  }
}

function resolveTranslations(t: CtxOrMaybeTranslations): Translations {
  return !t ? defaultTranslations : t.hasOwnProperty('translations') ? (t['translations'] || defaultTranslations) : t as Translations;
}


/** memoized `Jed` constructor */
const getJedIntance = memoize((translations: Translations) => new jed.Jed(translations));


/** minimal empty translations jed needs */
export const defaultTranslations: Translations = { "locale_data" : { "messages" : { "" : { "domain": "messages", "lang": "en", "plural_forms" : "nplurals=2; plural=(n != 1);" } } }, "domain" : "messages", "debug" : false }
