import { Eff } from '../eff';
import * as eff from '../eff';
import { HttpError } from '../http';
import * as http from '../http';
import * as t from '../decode';
import { success, failure } from '../either';


/// в отличии от других контейнеров, в `Eff` методы `map` и `chain` называются `mapEff` и `chainEff`
/// shim для nodejs
if (typeof(XMLHttpRequest) === 'undefined') global['XMLHttpRequest'] = require('xhr2');


/// декодер пытается привести значение к числу
const toNumber = t.decoder<number>('toNumber', value => {
  const result = Number(value);
  return isNaN(result) ? failure('cannot coerce to number') : success(result);
});


/// inner decoder
const channelDecoder = t.record({
  units: t.record({ distance: t.string, pressure: t.string, speed: t.string, temperature: t.string }),
  wind: t.record({ chill: toNumber, direction: toNumber, speed: toNumber }),
  atmosphere: t.record({ humidity: toNumber, pressure: toNumber, rising: toNumber, visibility: toNumber }),
  item: t.record({ condition: t.record({ date: t.string, temp: toNumber, text: t.string }) }),
});
type Channel = typeof channelDecoder['_a']


/// outer decoder
const resultsDecoder = t.at(['query', 'results', 'channel'], channelDecoder);


/// https://developer.yahoo.com/weather/
function yahooWeather(query: string): Eff<HttpError, Channel> {
  const q = `select units, wind, atmosphere, item.condition from weather.forecast where woeid in (select woeid from geo.places(1) where text="${query}")`;
  return http.send({    
    url: http.join('https://query.yahooapis.com/v1/public/yql', { q, format: 'json', env: 'store://datatables.org/Falltableswithkeys' }),
    method: 'GET',
  }).chainEff(http.expectJSON(resultsDecoder));
}


function formatResults(city: string, channel: Channel|null) {
  if (!channel) {
    console.log(`City: ${city}: Error`, );
    return;
  }
  const { atmosphere, wind } = channel;
  const { condition } = channel.item;
  const { distance, pressure, speed, temperature } = channel.units;
  console.log(`City: ${city}, ${condition.date}`);
  console.log(`\tCondition: ${condition.temp} ${temperature}, ${condition.text}`);
  console.log(`\tWind: ${wind.speed} ${speed}, direction: ${wind.direction}, chill: ${wind.chill} ${temperature}`);
  console.log(`\tAtmosphere: visibility: ${atmosphere.visibility} ${distance}, pressure: ${atmosphere.pressure} ${pressure}, humidity: ${atmosphere.humidity}, rising: ${atmosphere.rising}`);
}


const cities = ['Izhevsk', 'Moscow', 'London', 'Beijing'];
const cityGetWeather = city => yahooWeather(city).onError(() => eff.of(null)).mapEff(result => ({ city, result }));
const traverseCities = eff.traverse(cities, cityGetWeather);


/// вызов `subscribe` инициирует выполнение действий
traverseCities.subscribe(ethr => ethr.tag === 'Right' && ethr.value.forEach(({ city, result }, idx) => (idx !== 0 && console.log('------'), formatResults(city, result))));
