import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import withStyles, { WithStyles, StyleRules } from '@material-ui/core/styles/withStyles';
import { FieldProps } from './';
import * as moment from 'moment';
import TextField, { Props as TextFieldProps } from './TextField';
import { StandardProps, Popover } from '@material-ui/core';
import Icon from './Icon';
import 'react-dates/initialize';
import 'react-dates/lib/css/_datepicker.css';
import DayPicker from 'react-dates/lib/components/DayPicker';
import * as F from '~/fields';


// Props
export type Props = StandardProps<React.HTMLProps<HTMLDivElement>, ClassKey, 'value'> & FieldProps<moment.Moment|null> &  WithStyles<ClassKey> & {
  InputProps?: Partial<TextFieldProps>;
  format?: string;
}


// Model
export interface State {
  textFieldValue: string|null;
  open: boolean;
  value: null|string;
}


// Component
class MomentField extends React.Component<Props, State> {
  rootEl: HTMLElement|null;
  pickerEl: HTMLElement|null;
  state: State = { open: false, textFieldValue: null, value: null };
  unlisten: Function|null;

  projectContext = (input: FieldProps) => {
    const { classes } = this.props as Props & WithStyles;
    const self = this;

    const endAdornment = <React.Fragment>
      <Icon data-ignore-focus-in className={classes.iconDate} onClick={this.handleVisibilityToggle}>date_range</Icon>
      <Icon data-ignore-focus-in className={classes.iconClose} onClick={this.handleClear}>close</Icon>
    </React.Fragment>;

    return {
      ...input,
      endAdornment,
      value: inputValue(this.state, input),
      onValueChange: this.handleChange,
      onBlur: this.handleBlur,
      onKeyDown: this.handleKeyDown,
    };

    function inputValue(state: State, input: FieldProps) {
      return state.value !== null ? state.value : input.value ? input.value.format(self.format(self.props)) : '';
    }
  };

  componentDidMount() {
    document.addEventListener('focusin', this.handleFocusIn);
    document.addEventListener('click', this.handleFocusIn);
    this.unlisten = () => (document.removeEventListener('focusin', this.handleFocusIn), document.removeEventListener('click', this.handleFocusIn), this.unlisten = null);
  }

  componentWillUnmount() {
    this.unlisten && this.unlisten();
  }

  handleRootRef: React.Ref<HTMLDivElement> = c => {
    if (!c) { this.rootEl = null; return; }
    const rootEl = ReactDOM.findDOMNode(c); if (!rootEl) return;
    this.rootEl = rootEl as HTMLElement;
  };

  handlePickerRef: React.Ref<HTMLDivElement> = c => {
    if (!c) { this.pickerEl = null; return; }
    const rootEl = ReactDOM.findDOMNode(c); if (!rootEl) return;
    this.pickerEl = rootEl as HTMLElement;
  }

  handleChange: TextFieldProps['onValueChange'] = value => {
    this.setState({ value });
  };

  handleBlur = () => {
    const { onValueChange } = this.props;
    const { value } = this.state;
    if (!value) return;
    const nextValue = moment(value, this.format(this.props)); if (!nextValue.isValid()) return;
    this.setState({ value: null })
    const prevValueOf = this.props.value ? this.props.value.valueOf() : 0;
    const nextValueOf = nextValue.valueOf();
    prevValueOf !== nextValueOf && onValueChange && onValueChange(nextValue);
  };

  handleKeyDown: TextFieldProps['onKeyDown'] = e => {
    switch (e.key) {
      case 'Enter':
        this.handleBlur();
        break;
    }
  };

  handleFocusIn = (e: FocusEvent) => {
    const { rootEl, pickerEl } = this;
    const { disabled } = this.props; if (disabled) return;
    const targetElement = e.target as Element;
    if (targetElement.hasAttribute('data-ignore-focus-in')) return;
    const open = isElementInside(targetElement);
    this.setState({ open });

    function isElementInside(node: Node): boolean {
      if (node === rootEl || node === pickerEl) return true;
      return node.parentNode ? isElementInside(node.parentNode) : false;
    }
  };

  handleClear = (e) => {
    e.preventDefault(); e.stopPropagation();
    const { onValueChange } = this.props;
    onValueChange && onValueChange(null);
  };

  handleVisibilityToggle = (e: React.MouseEvent<HTMLSpanElement>) => {
    const { open } = this.state;
    this.setState({ open: !open });
  };

  handleClose = (e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (this.state.open) this.setState({ open: false });
  };

  handleDatesChange = (date: moment.Moment) => {
    const { value, onValueChange } = this.props;
    const prev = value || moment().startOf('minute');
    const next = date.clone();
    next.hours(prev.hours());
    next.minutes(prev.minutes());
    onValueChange && onValueChange(next);
  };

  format(props: Props): string {
    return props.format || 'DD/MM/YYYY';
  }

  render() {
    const { classes, ctx, error, InputProps, onBlur, onChange, onFocus, value, format: formapProps, disabled, ...rest } = this.props;
    const { rootEl } = this;
    const { open } = this.state;
    const rootClass = classes.root;

    return <F.Modifier proj={this.projectContext}>
      <div {...rest} className={rootClass} ref={this.handleRootRef}>
        <TextField
          {...InputProps as any}
          disabled={disabled}
          error={error}
          className={classes.input}
        />
        {rootEl && <Popover className={classes.modal} anchorEl={rootEl} open={open} onClose={this.handleClose} anchorOrigin={{ vertical: 'bottom', horizontal: 'left' }} disableAutoFocus disableEnforceFocus disableRestoreFocus hideBackdrop>
          <DayPicker
            ref={this.handlePickerRef}
            date={value}
            onDayClick={this.handleDatesChange}
            numberOfMonths={1}
          />
        </Popover>}
      </div>
    </F.Modifier>;
  }
}

export default withStyles(styles)(MomentField);


// CSS классы
export type ClassKey = 'root'|'modal'|'input'|'iconClose'|'iconDate';


// Styles
export function styles(theme: Theme): StyleRules<ClassKey> {
  const { unit } = theme.spacing;

  return {
    root: {
    },

    iconClose: {
      position: 'absolute',
      right: unit,
    },

    iconDate: {
      position: 'absolute',
      right: unit * 4,
    },

    modal: {
      /* width: 0,
       * height: 0, */
      /* top: unit,
       * left: -unit,*/
    },

    input: {
      width: 180,
      paddingRight: `${unit * 8}px !important`,
      position: 'relative',
    },
  };
}
