import { Decoder } from '../decoder';
import * as t from '../decoder';
import { Either } from '../either';
import * as either from '../either';


/** a source to read config from */
export type ConfigSource =
  | { tag: 'Cli', prefix: string, args: string[] }
  | { tag: 'Location/search', value: string, transformKey(x: string): string }
  | { tag: 'Config', value: Record<string, any> }


/** a decoder together with a function that reads a string into an arbitrary value (string is usually from get-parameters) */
export class ConfigReader<A> {
  readonly _A: A;

  constructor(
    readonly decoder: Decoder<A>,
    readonly fromString?: (x: string) => any,
  ) {}

  /** provide default value */
  withDefault<B>(a: B): ConfigReader<A|B> {
    return new ConfigReader(this.decoder.withDefault(a), this.fromString);
  }
  
  /** apply pure function */
  map<B>(f: (x: A) => B): ConfigReader<B> { return new ConfigReader(this.decoder.map(f), this.fromString); }

}


/** type alias */
export type ConfigRecord = Record<string, ConfigReader<any>>;


/** a source to read config from */
type ConfigSourceProcessed =
  | { tag: 'Cli', value: Record<string, string> }
  | { tag: 'Location/search', value: Record<string, string> }
  | { tag: 'Config', value: Record<string, any> }


/** fold configs */
function foldConfigs<R extends ConfigRecord>(sources: ConfigSource[], config: R): Either<{ output: Partial<{ [K in keyof R]: R[K]['_A']}>, problem: t.Problem }, { [K in keyof R]: R[K]['_A'] }> {

  const processedSources = sources.map(function (source: ConfigSource): ConfigSourceProcessed {
    switch (source.tag) {
      case 'Cli': {
	const value: Record<string, any> = {};
	for (let i = 0; i < source.args.length; i++) {
	  const prefixLen = '--'.length + source.prefix.length;
	  const rest = source.args[i].substr(prefixLen);
	  const [k, v] = rest.indexOf('=') === -1 ? [rest, source.args[++i] || ''] : source.args[i].split('=');
	  const cfg = config[k];
	  if (config.hasOwnProperty(k) && cfg.fromString) value[k] = cfg.fromString(v);
	}
	return { tag: 'Cli', value };
      }
      case 'Location/search': {
	const substring = source.value[0] === '?' ? source.value.substr(1) : source.value;
	const value = substring.split('&').map(x => x.split('=')).reduce((acc, [k, v]: any) => (acc[source.transformKey(decodeURIComponent(k))] = v ? decodeURIComponent(v) : '', acc), {} as any);
	return { tag: 'Location/search', value };
      }
      case 'Config': return source;
    }
  });
  
  /** fold all sources into one record */
  const record: Record<string, any> = {};
  for (let key in config) {
    for (let i in processedSources) {
      const source = processedSources[i];
      if (source.tag === 'Cli' && source.value.hasOwnProperty(key)) {
	record[key] = source.value[key];
	break;
      }
      if (source.tag === 'Location/search' && source.value.hasOwnProperty(key)) {
	const cfg = config[key];
	cfg.fromString && (record[key] = cfg.fromString(source.value[key]));
	break;
      }
      if (source.tag === 'Config' && source.value.hasOwnProperty(key)) {
	record[key] = source.value[key];
	break;
      }
    }    
  }
  
  const output = {} as { [K in keyof R]: R[K]['_A']};
  for (let key in config) {
    if (!config.hasOwnProperty(key)) continue;
    const ethr = config[key].decoder.validate(record[key]);
    switch(ethr.tag) {
      case 'Left': { ethr.value.path.push(key); return either.failure({ output, problem: ethr.value }); }
      case 'Right': output[key] = ethr.value; break;
    }
  }
  
  return either.success(output);
}
      

/** read config */
export function read<R extends Record<string, ConfigReader<any>>>(sources: ConfigSource[], config: R): Either<t.Problem, { [K in keyof R]: R[K]['_A']}> {
  const ethr = foldConfigs(sources, config);
  return ethr.tag === 'Right' ? ethr as any : either.failure(ethr.value.problem);
}


/** primitives */
export const booleanReader = new ConfigReader(t.boolean, x => { if (x === 'false') return false; if (x === '0') return false; return true; });
export const floatReader = new ConfigReader(t.float, parseFloat);
export const intReader = new ConfigReader(t.int, parseInt);
export const natReader = new ConfigReader(t.nat, parseInt);
export const stringReader = new ConfigReader(t.string, x => x);
export const anyReader = new ConfigReader(t.any, x => x);

export function array<T>(cr: ConfigReader<T>): ConfigReader<T[]> {
  return new ConfigReader(t.array(cr.decoder));
}


/** constructor */
export function reader<a>(fromString: (x: string) => any, decoder: Decoder<a>): ConfigReader<a> {
  return new ConfigReader(decoder, fromString);
}


/** export primitives */
export { booleanReader as boolean, floatReader as float, intReader as int, natReader as nat, stringReader as string, anyReader as any };
