import * as React from 'react';
import { isEqual } from 'lodash';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import withStyles, { WithStyles, StyleRules } from '@material-ui/core/styles/withStyles';
import * as classNames from 'classnames';
import { FieldProps } from './';
import { MenuItem, StandardProps } from '@material-ui/core';
import Select, { SelectProps } from '@material-ui/core/Select';
import { AuthCtx as Ctx } from '~/context';
import { Err, notifyError } from '~/context';
import * as Rx from 'rxjs';
import { Eff } from '@bitmaster/core';


// Источник для опций
export type Source<A> =
  | Array<A>
  | AsyncSource<A>


// Асинхронные автодополенния
export type AsyncSource<A> = {
  getCollection(ctx: Ctx): Eff<Err, A[]>;
}


// Props
export type Props<A = any> = StandardProps<React.HTMLProps<HTMLDivElement>, ClassKey> & FieldProps<A> & {
  selectProps?: Partial<SelectProps>;
  source: Source<A>;
  renderItem?(item: A): string;
  prepareItem?(item: A): any;
  isEqual?(a: A, b: A): boolean;
  nullCase?: string;
}


// State
export interface State<A> {
  open: boolean;
  options: null|'pending'|A[];
}


// Тип для запроса коллекции
export interface SelectFieldQuery {
  offset: number;
  limit: number;
}


// Component
// @ts-ignore
@withStyles(styles)
export default class SelectField<A = any> extends React.Component<Props<A>, State<A>> {
  state: State<A> = { open: false, options: null };
  subscription: Rx.Subscription|null = null;

  componentDidMount() {
    const { source, ctx } = this.props;
    if (!Array.isArray(source) && ctx) {
      this.fetchOptions(ctx, source, false);
    }
  }

  componentWillUnmount() {
    if (this.subscription) {
      this.subscription.unsubscribe();
      this.subscription = null;
    }
  }

  handleChange: SelectProps['onChange'] = e => {
    const { onValueChange, prepareItem } = this.props;
    const options = this.getOptions();
    const idx = Number(e.target.value); if (isNaN(idx) || options.length <= idx) return;
    if (idx < 0) {
      onValueChange && onValueChange(null as any);
      return;
    }
    const prepare = prepareItem || (x => x);
    onValueChange && onValueChange(prepare(options[idx]));
  };

  handleOpen = () => {
    const { source, ctx } = this.props;
    if (Array.isArray(source)) {
      this.setState({ open: true });
    }
    else if (ctx) {
      this.fetchOptions(ctx, source);
    }
  };

  handleClose = () => {
    this.setState({ open: false });
  };

  getOptions() {
    return Array.isArray(this.state.options) ? this.state.options : Array.isArray(this.props.source) ? this.props.source : [];
  }

  fetchOptions(ctx: Ctx, source: AsyncSource<A>, open=true) {
    if (this.subscription) return;
    this.setState({ options: 'pending' });
    this.subscription = source.getCollection(ctx).subscribe(ethr => {
      this.subscription = null;
      if (ethr.tag === 'Left') {
        notifyError(ethr.value);
        this.setState({ options: null });
      } else {
        this.setState({ options: ethr.value, open });
      }
    });
  }

  render() {
    const { classes, error, nullCase, /*dirty, selectProps, options: optionsProps, */ disabled, renderItem, /*onFocus, */value, className, isEqual: isEqualProp /*...rest*/ } = this.props as Props<A> & WithStyles<ClassKey>;
    const { open } = this.state;
    const rootClass = classNames(className, classes.root, {
      [classes.error]: error,
    });
    const options = this.getOptions();
    const predicate = isEqualProp || isEqual;
    const valueIdx = options.findIndex(x => predicate(x, value!));

    return (
      <Select value={valueIdx} onChange={this.handleChange} className={rootClass} disabled={disabled} open={open} onOpen={this.handleOpen} onClose={this.handleClose}>
        {nullCase && <MenuItem key="@null" value={-1}>{nullCase}</MenuItem>}
        {options.map((item, idx) => <MenuItem key={idx} value={idx}>{(renderItem || String)(item)}</MenuItem>)}
      </Select>
    );
  }
}


// CSS классы
export type ClassKey = 'root'|'error';


// Styles
function styles(theme: Theme): StyleRules<ClassKey> {
  const { unit } = theme.spacing;

  return {
    root: {
      '&:before': {
        display: 'none',
      },
      '& > div > div': {
        paddingLeft: unit,
        background: `rgba(0,0,0,0.04)`,
        borderRadius: 2,
      },
    },

    error: {
      background: `rgba(255,0,0,0.08)`,
    },
  };
}
