import * as React from 'react';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import withStyles, { StyleRules, WithStyles } from '@material-ui/core/styles/withStyles';
import * as classNames from 'classnames';
import { FieldProps } from './';
import { StandardProps } from '@material-ui/core';
import { withGettext, Gettext, LocaleCtx } from '~/gettext';
const eye = require('./eye.svg');
const eyeSlash = require('./eye-slash.svg');


// Props
export type Props = StandardProps<React.HTMLProps<HTMLDivElement>, ClassKey, 'ref'|'onKeyDown'|'disabled'> & FieldProps<string> & WithStyles<ClassKey> & {
  __: Gettext;
  ctx?: LocaleCtx;
  type?: React.HTMLProps<HTMLInputElement>['type'];
  filter?(s: string): boolean;
  regexFilter?: RegExp;
  inputProps?: React.HTMLProps<HTMLInputElement>;
  beginAdornment?: React.ReactNode;
  endAdornment?: React.ReactNode;
  Input?: React.ComponentType|string;
  placeholder?: string;
  fullWidth?: boolean;
};


// State
export type State = {
  type: string|null;
};


// Component
export class TextField extends React.Component<Props, State> {
  state: State = { type: null };
  
  static getDerivedStateFromProps(props: Props, state: State) {
    if (props.type === 'password-switch' && state.type === null) return { type: 'password' };
    if (props.type !== 'password-switch' && state.type !== null) return { type: null };
    return null;
  }
  
  handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { onValueChange, filter, regexFilter } = this.props;
    if (filter && !filter(e.target.value)) return;
    if (regexFilter && !regexFilter.test(e.target.value)) return;
    onValueChange && onValueChange(e.target.value);
  };

  handleSwitchPassword = () => {
    const { type } = this.state;
    this.setState({ type: type === 'password' ? 'text' : 'password' })
  };
  
  render() {
    const {
      __,
      ctx,
      classes,
      //      dirty,
      fullWidth,
      error,
      inputProps,
      onBlur,
      onChange,
      onFocus,
      value,
      beginAdornment,
      endAdornment: _endAdornment,
      className,
      disabled,
      onKeyDown,
      type,
      Input: InputProp,
      placeholder,
      ...rest
    } = this.props;

    const rootClass = classNames(className, classes.root, {
      [classes.disabled]: disabled,
      [classes.error]: error,
      [classes.textarea]: InputProp === 'textarea',
      [classes.fullWidth]: fullWidth,
    });
    const inputClass = classNames(classes.input, {
      [classes.error]: error,
    });
    const Input: any = InputProp || 'input';
    const endAdornment = type !== 'password-switch' ? _endAdornment : <React.Fragment>
      {_endAdornment}
      <span className={classes.eye} onClick={this.handleSwitchPassword} dangerouslySetInnerHTML={{ __html: this.state.type === 'password' ? eye : eyeSlash}} data-rh={this.state.type === 'password' ? __('Show password') : __('Hide password')}/>
    </React.Fragment>;

    return <div {...rest} className={rootClass}>
      {beginAdornment}
      <Input
        placeholder={placeholder}
        {...inputProps}
        type={type === 'password-switch' ? this.state.type : type}
        className={inputClass}
        onChange={this.handleChange}
        value={value || ''}
        disabled={disabled}
        onFocus={onFocus}
        onBlur={onBlur}
        onKeyDown={onKeyDown}
        data-rh-at="right"
        data-rh={typeof(error) === 'string' || typeof(error) === 'function' ? typeof(error) === 'string' ? error : error(ctx) : undefined}
      />
      {endAdornment}
    </div>;
  }
}

export default withStyles(styles)(withGettext(require('./i18n'))(TextField));


// CSS классы
export type ClassKey = 'root'|'error'|'input'|'textarea'|'eye'|'fullWidth'|'disabled';


// Styles
function styles(theme: Theme): StyleRules<ClassKey> {
  const { unit } = theme.spacing;
  
  return {
    root: {
      height: unit * 4.5,
      boxSizing: 'border-box',
      display: 'inline-flex',
      alignItems: 'center',
      background: `rgba(0,0,0,0.04)`,
      borderRadius: 3,
      position: 'relative',
      '&$error': {
        background: `rgba(255,0,0,0.08)`,
      },
    },
    
    disabled: {
      cursor: 'not-allowed',
    },

    input: {
      padding: [unit, unit] as any,
      outline: 'none',
      border: 'none',
      background: 'none',
      flex: '10 10 auto',
      height: '100%',
      width: '100%',
      boxSizing: 'border-box',
      fontSize: 18,
      color: theme.palette.text.primary,
      '&[disabled]': {
        color: theme.palette.text.disabled,
        cursor: 'not-allowed',
      },
      '&:after': {
        content: '""',
        position: 'absolute',
        display: 'block',
        width: '100%',
        height: 0,
      },
      '&:focus:after': {
        background: theme.palette.primary.main,
        height: 2,
      },
      '&$error:after': {
        background: theme.palette.error.main,
      },
      '& + *': {
        marginLeft: -unit,
      },
      '&::placeholder': {
        fontStyle: 'italic',
      },
      '&$error::placeholder': {
        color: `rgba(255,0,0,0.54)`,
      },
    },
    
    textarea: {
      height: 'auto',
      width: '100%',
      padding: 0,
      '& textarea': { padding: unit },
    },

    eye: {
      width: 22,
    },

    error: {
    },

    fullWidth: {
      width: '100%',
    },
  };
}

