import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { memoize } from 'lodash';
import MenuList from '@material-ui/core/MenuList';
import MenuItem, { MenuItemProps } from '@material-ui/core/MenuItem';
import Modal from '@material-ui/core/Modal';
import Paper from '@material-ui/core/Paper';
import withStyles, { WithStyles, StyleRules } from '@material-ui/core/styles/withStyles';
import { StandardProps } from '@material-ui/core';
import PendingOverlay from '@bitmaster/components/PendingOverlay';
import { LocaleCtx, Gettext, withGettext } from '~/gettext';


// props
export type Props<A=any> = StandardProps<React.HTMLProps<HTMLDivElement>, string> & WithStyles<string> & {
  __: Gettext;
  ctx: LocaleCtx;
  suggestions: A[];
  renderSuggestion?(a: A): React.ReactElement<MenuItemProps>|string;
  anchorEl?: HTMLElement;
  open?: boolean;
  marginThreshold?: number;
  pending?: boolean;
  dontMove?: boolean;

  // callbacks
  onSelect?(a: A): void;
  onVisibilityChange?(open: boolean): void;
}


// state
export interface State {
  selected: number;
}


// component
class Suggestions<A> extends React.Component<Props<A>, State> {
  state = { selected: 0 };
  paperEl: HTMLElement|null = null;
  unlisten: Function|null = null;

  componentDidMount() {
    if (this.props.open) this.listen();
  }

  componentWillUnmount() {
    this.unlisten && this.unlisten();
  }

  componentWillReceiveProps(nextProps: Props) {
    if (nextProps.open && !this.props.open) { // show component
      this.listen();
    }
    if (!nextProps.open && this.props.open) { // hide component
      this.setHovered(0);
      this.unlisten && this.unlisten();
    }
    if (nextProps.suggestions !== this.props.suggestions) {
      this.setHovered(0);
    }
  }

  componentDidUpdate(prevProps: Props) {
    const dontMove = typeof(this.props.dontMove) === 'boolean' ? this.props.dontMove : true;
    if (prevProps.suggestions !== this.props.suggestions && !dontMove && this.paperEl) {
      this.handlePaperRef(this.paperEl as any);
    }
  }

  listen() {
    document.addEventListener('keydown', this.handleKeyDown);
    document.addEventListener('focusin', this.handleFocusClick);
    document.addEventListener('click', this.handleFocusClick);
    this.unlisten = () => {
      document.removeEventListener('keydown', this.handleKeyDown);
      document.removeEventListener('focusin', this.handleFocusClick);
      document.removeEventListener('click', this.handleFocusClick);
      this.unlisten = null;
    };
  }

  handleFocusClick = (e: FocusEvent|MouseEvent) => {
    const { paperEl, props: { anchorEl, onVisibilityChange } } = this;
    onVisibilityChange && onVisibilityChange(isElementInside(e.target as Node));

    function isElementInside(node: Node): boolean {
      if (node === anchorEl || node === paperEl) return true;
      return node.parentNode ? isElementInside(node.parentNode) : false;
    }
  };

  setHovered(selected: number) {
    const SCROLL_BOTTOM_THRESHOLD = 0.8;
    const SCROLL_TOP_THRESHOLD = 0.1;

    this.setState({ selected }, () => {
      if (!this.paperEl) return;
      //const menuItemClasses = styleManager.render(menuItem.styleSheet);
      const selected = this.paperEl.querySelector('[data-is-selected=true]') as HTMLElement | null; if (!selected) return;
      const menu = selected.parentNode as HTMLElement;
      const menuRect = menu.getBoundingClientRect();
      if (selected.offsetTop - menu.scrollTop >= SCROLL_BOTTOM_THRESHOLD * menuRect.height) {
        // reached the point where we need to scroll down
        menu.scrollTop = selected.offsetTop - SCROLL_BOTTOM_THRESHOLD * menuRect.height;
      }
      else if (selected.offsetTop - menu.scrollTop < SCROLL_TOP_THRESHOLD * menuRect.height) {
        // need to scroll up
        menu.scrollTop = selected.offsetTop - SCROLL_TOP_THRESHOLD * menuRect.height;
      }
    });
  }

  handleKeyDown = (e: KeyboardEvent) => {
    const { suggestions, onSelect } = this.props;

    switch (e.key) {
      case 'ArrowDown': 
        if (this.state.selected < suggestions.length - 1)
          this.setHovered(this.state.selected + 1);
        break;
      case 'ArrowUp': 
        if (this.state.selected > 0)
          this.setHovered(this.state.selected - 1);
        break;
      case 'Enter': {
        e.stopPropagation();
        e.preventDefault();
        if (this.state.selected >= suggestions.length) break;
        onSelect && onSelect(suggestions[this.state.selected]);
        break;
      }
    }
  };

  handleSuggestionClick = memoize((idx: number) => (e: any) => {
    const { suggestions, onSelect } = this.props;
    e.stopPropagation();
    e.preventDefault();
    if (idx >= suggestions.length) return;
    onSelect && onSelect(suggestions[idx]);
  });

  handleClose = () => {
    const { onVisibilityChange } = this.props;
    onVisibilityChange && onVisibilityChange(false);
  };

  handlePaperRef = (c: React.Component|null) => {
    const { anchorEl } = this.props;
    if (!c || !anchorEl) return;
    this.paperEl = ReactDOM.findDOMNode(c) as HTMLElement;
    const marginThreshold = this.props.marginThreshold || 8;
    const heightThreshold = window.innerHeight - marginThreshold;
    const paperRect = this.paperEl.getBoundingClientRect();
    const anchorRect = anchorEl.getBoundingClientRect();
    const rectHeight = paperRect.height;
    let top = anchorRect.bottom + 4;
    const bottom = top + rectHeight
    if (bottom > heightThreshold && anchorRect.top - window.scrollY - rectHeight - 8 > marginThreshold) {
      top = anchorRect.top - 8 - rectHeight;
    }
    this.paperEl.style.top = top + 'px';
    this.paperEl.style.left = anchorRect.left + 'px';
    this.paperEl.style.minWidth = anchorRect.width + 'px';
  };

  // Render suggestion list
  renderSuggestions(suggestions: Props['suggestions']) {
    const { renderSuggestion } = this.props;
    const { selected } = this.state;
    return suggestions.map((x, idx) => {
      const element = renderSuggestion ? renderSuggestion(x) : String(x);
      return React.isValidElement(element)
           ? React.cloneElement(element, { key: idx, onMouseDown: e => { this.handleSuggestionClick(idx)(e); element.props.onClick && element.props.onClick(e); } , selected: selected === idx, 'data-is-selected': selected === idx } as MenuItemProps)
           : <MenuItem key={idx} selected={selected === idx} onMouseDown={this.handleSuggestionClick(idx)} data-is-selected={selected === idx}>{element}</MenuItem>;
    });
  }

  render() {
    const { __, classes, open, pending, suggestions } = this.props;
    const PaperProps = { ref: this.handlePaperRef, className: classes.paper };
    
    return <Modal className={classes.modal} open={!!open} onClose={this.handleClose} disableAutoFocus disableEnforceFocus disableRestoreFocus hideBackdrop>
      <Paper {...PaperProps}>
        {!!suggestions.length && <MenuList className={classes.menu}>{this.renderSuggestions(suggestions)}</MenuList>}
        {!suggestions.length && <div className={classes.empty}><span>{__('Nothing found…')}</span></div>}
        <PendingOverlay pending={pending || false}/>
      </Paper>
    </Modal>;
  }
}

export default withStyles(styles)(withGettext(require('./i18n'))(Suggestions));


// Styles
export function styles(theme): StyleRules {
  const { unit } = theme.spacing;
  
  return {
    paper: {
      position: 'absolute',
      zIndex: 9000,
    },

    menu: {
      maxHeight: 300,
      overflowY: 'auto',
    },

    modal: {
      width: 0,
      height: 0,
    },

    empty: {
      color: theme.palette.text.secondary,
      padding: [unit, unit * 2],
      textAlign: 'center',
      fontStyle: 'italic',
      width: '100%',
    },
  };
}
