import { Err, AuthCtx } from "~/context";
import { Cmd, noop, Eff, cmd } from "../../core";


/**
 * Экшен для управления флагом загрузки
 */
export type PendingAction =
  | { tag: '@Pending', pending: boolean }


// Pending
export function pending<A>(command: Cmd<A>): Cmd<A> {
  return cmd.concat(
    cmd.of({ tag: '@Pending', pending: true } as never),
    command,
    cmd.of({ tag: '@Pending', pending: false } as never),
  );
}


// Контекст
export type Ctx<A> = (AuthCtx | {}) & {
  options: Options<A>;
};


// Options
export type Options<A> = {
  source: Source<A>;
  printItem?(x: A): string;
  limit?: number;
  openOnFocus?: boolean;
  openOnClick?: boolean;
};


// Параметры для `pageCollection`
export interface AutoCompleteQuery {
  offset: number;
  limit: number;
  search: string
}
export const defaultQuery = { offset: 0, search: '', limit: 20 };


// Источник для автодополнений
export type Source<A> =
  | Array<A>
  | AsyncSource<A>
  | AsyncSource2<A>


// Асинхронные автодополенния
export type AsyncSource<A> = {
  pageCollection(ctx: AuthCtx, query: AutoCompleteQuery): Eff<Err, [A[], number]>;
}

// Асинхронные автодополенния
export type AsyncSource2<A> = {
  getCollection(ctx: AuthCtx, query: AutoCompleteQuery): Eff<Err, A[]>;
}


// Actions
export type Action<A> =
  | PendingAction
  | { tag: 'Error', error: Err }
  | { tag: 'Search', value: string }
  | { tag: 'Offset', value: number }
  | { tag: 'More' }
  | { tag: 'Query', query: AutoCompleteQuery }
  | { tag: 'Query/success', suggestions: A[], total: number, query: AutoCompleteQuery }
  | { tag: 'Focus' }
  | { tag: 'Blur' }
  | { tag: 'Close' }
  | { tag: 'Open' }
  | { tag: 'Click' }


// State
export type State<A> = {
  open: boolean;
  suggestions: A[];
  pending: boolean;
  total: number;
  query: AutoCompleteQuery;
};


// Init
export function init<A>(): State<A> {
  return { open: false, suggestions: [], pending: false, total: 0, query: defaultQuery };
}


// Update
export function update<A>(ctx: Ctx<A>, action: Action<A>, state: State<A>): [State<A>, Cmd<Action<A>>] {
  const onFailure = (error: Err) => ({ tag: 'Error', error } as Action<A>);
  switch(action.tag) {
    case '@Pending': {
      return [{ ...state, pending: action.pending }, noop];
    }
    case 'Error': {
      return [state, noop];
    }
    case 'Query': {
      const { query } = action;
      const { options: { source } } = ctx;
      const printItem = ctx.options.printItem || (x => JSON.stringify(x));
      
      if (Array.isArray(source)) {
        // Поиск подходящих записей в массиве
        const suggestions: A[] = [];
        let skipped = 0;
        for (let i = 0; i < source.length && suggestions.length < query.limit; i++) {
          if (!query.search || fuzzyFilter(query.search, printItem(source[i]))) {
            if (query.offset > skipped) {
              skipped++;
              continue;
            }
            suggestions.push(source[i]);
          }
        }
        return [{ ...state, suggestions, query, open: true, total: source.length }, noop];
      } else if ('pageCollection' in source) {
        if (!('auth' in ctx) || !ctx.auth) return [state, noop];
        // Запрос на поиск записей
        const onSuccess = ([suggestions, total]) => ({ tag: 'Query/success', suggestions, total, query } as Action<A>);
        const command = source.pageCollection(ctx, query).perform(onFailure, onSuccess);
        return [{ ...state, open: true }, pending(command)];        
      } else if ('getCollection' in source) {
        if (!('auth' in ctx) || !ctx.auth) return [state, noop];
        // Запрос на поиск записей
        const onSuccess = (suggestions) => ({ tag: 'Query/success', suggestions, total: 0, query } as Action<A>);
        const command = source.getCollection(ctx, query).perform(onFailure, onSuccess);
        return [{ ...state, open: true }, pending(command)];        
      } else {
        throw new Error('Unknown source');
      }
    }
    case 'Query/success': {
      const { suggestions, total, query } = action;
      return [{ ...state, suggestions, total, query }, noop];
    }
    case 'Offset': {
      return [state, cmd.of<Action<A>>({ tag: 'Query', query: { ...state.query, offset: action.value } })];
    }
    case 'Search': {
      return [state, cmd.of<Action<A>>({ tag: 'Query', query: { ...defaultQuery, search: action.value } })];
    }
    case 'More': {
      const { query: { limit } } = state;
      return [state, cmd.of<Action<A>>({ tag: 'Query', query: { ...state.query, limit: limit + 20 } })];
    }
    case 'Focus': {
      if (ctx.options.openOnFocus && !ctx.options.openOnClick) {
        // Не открываем попап на фокусе если указан
        // `ctx.options.openOnClick`, тк он откроется на клике
        return [state, cmd.of<Action<A>>({ tag: 'Query', query: state.query })];
      }
      return [state, noop];
    }
    case 'Blur': {
      return [{ ...state, open: false }, noop];
    }
    case 'Close': {
      return [{ ...state, open: false }, noop];
    }
    case 'Open': {
      if (state.open) return [state, noop];
      return [state, cmd.of<Action<A>>({ tag: 'Query', query: defaultQuery })];
    }
    case 'Click': {
      if (ctx.options.openOnClick) {
        if (state.open) return [{ ...state, open: false }, noop];
        return [state, cmd.of<Action<A>>({ tag: 'Query', query: defaultQuery })];
      }
      return [state, noop];
    }
  }
}


// https://github.com/callemall/material-ui/blob/a96ddc1b45b0fa325f0ec2d0cbd26ed565c9b40c/src/AutoComplete/AutoComplete.js#L599
export function fuzzyFilter(searchText: string, x: string): boolean {
  const compareString = x.toLowerCase();
  searchText = searchText.toLowerCase();

  let searchTextIndex = 0;
  for (let index = 0; index < x.length; index++) {
    if (compareString[index] === searchText[searchTextIndex]) {
      searchTextIndex += 1;
    }
  }
  return searchTextIndex === searchText.length;
}
