import { Expr } from './internal/expr';


/** adt */
export type Option<A> = None<A> | Some<A>
export type Maybe<A> = Option<A>


/** instance method for convenience */
export class OptionBase<A> {
  readonly _A: A;

  /** map */
  map<B>(f: (a: A) => B): Option<B> {
    const self = this as any as Option<A>;
    switch (self.tag) {
      case 'None': return none;
      case 'Some': return new Some(f(self.value));
    }
  }
  
  /** map */
  mapTo<B>(value: B): Option<B> {
    const self = this as any as Option<A>;
    switch (self.tag) {
      case 'None': return none;
      case 'Some': return some(value);
    }
  }

  /** chain */
  chain<B>(f: (a: A) => Option<B>): Option<B> {
    const self = this as any as Option<A>;
    switch (self.tag) {
      case 'None': return none;
      case 'Some': return f(self.value);
    }
  }

  /** chain */
  chainTo<B>(value: Option<B>): Option<B> {
    const self = this as any as Option<A>;
    switch (self.tag) {
      case 'None': return none;
      case 'Some': return value;
    }
  }

  /** unwrap */
  unwrap<B extends Expr, C extends Expr>(fromNone: B, fromSome: (x: A) => C): B|C {
    const self = this as any as Option<A>;
    switch (self.tag) {
      case 'None': return fromNone;
      case 'Some': return fromSome(self.value);
    }
  }

  /** withDefault */
  withDefault<B extends Expr>(def: B): A|B {
    return this.unwrap(def, x => x);
  }  
}

  
/** empty container */
export class None<A> extends OptionBase<A> {
  readonly _a: A;
  readonly tag: 'None' = 'None';
}


/** container with a value */
export class Some<A> extends OptionBase<A> {
  readonly _a: A;
  readonly tag: 'Some' = 'Some';
  constructor(
    readonly value: A,
  ) { super(); }
}


/** traverse an array */
export function traverse<A,B>(arr: Array<A>, f: (a: A, idx: number) => Option<B>): Option<B[]> {
  const output = [] as B[];
  for (let i = 0; i < arr.length; i++) {
    const option = f(arr[i], i);
    switch (option.tag) {
      case 'None': return option as any;
      case 'Some': output.push(option.value); break;
    }
  }
  return new Some(output);
}


/** aliases */
export const none = new None<never>();
export function some<A extends Expr>(a: A): Option<A> { return new Some<A>(a); }
export { some as of };
