import { HasEffect, Subscribe, Eff } from '../eff';
import * as eff from '../eff';
import { Decoder, Problem } from '../decoder';
import { Either } from '../either';
import * as either from '../either';


/** http method */
export type Method = 'GET'|'POST'|'PUT'|'DELETE'|'PATCH';


/** request */
export interface Request {
  url: string;
  method: Method;
  body?: any;
  headers?: Record<string, string|number|undefined|null>;
  withCredentials?: boolean;
  timeout?: number;
}

export interface RequestProgress extends Request {
  progress: true;
}


/** Raw error */
export type HttpError =
  | { tag: 'BadUrl', desc: string }
  | { tag: 'BadPayload', desc: string }
  | { tag: 'ValidationProblem', problem: Problem, url: string }
  | { tag: 'BadStatus', status: number, desc: string }
  | { tag: 'Timeout' }
  | { tag: 'NetworkError' }


/** Responce */
export interface Response {
  url: string;
  status: number;
  statusText: string;
  headers: Record<string, string>;
  body: string;
}


/** Query params */
export type ParamsPrimitive = number|string|undefined|null;
export type Params = Record<string, ParamsPrimitive|ParamsPrimitive[]>;


/** Progress */
export type Progress =
  | { tag: 'Computable', total: number, loaded: number }
  | { tag: 'Uncomputable' }


export class HttpEffect<A> extends HasEffect<HttpError, A> {
  constructor(
    readonly request: Request|RequestProgress,
  ) { super(); };

  toEffect() {
    return new Subscribe<HttpError, any>((onNext, onComplete) => doXHR(this.request, onNext, onComplete));
  }
}


/** send a request */
export function send(req: Request): HttpEffect<Response>;
export function send(req: Request|RequestProgress): HttpEffect<unknown> {
  return new HttpEffect(req);
}


/** shortcut for GET requests */
//export function get(url: string, request?: Omit<RequestProgress, 'url'|'method'>): HttpEffect<Either<Progress, Response>>;
export function get(url: string, request?: Omit<Request, 'url'|'method'>): Eff<HttpError, Response>;
export function get(url: string, request?: Omit<Request|RequestProgress, 'url'|'method'>) {
  return send({ ...request, method: 'GET', url });
}


/** shortcut for POST requests */
export function post(url: string, request?: Omit<RequestProgress, 'url'|'method'>): HttpEffect<Either<Progress, Response>>;
export function post(url: string, request?: Omit<Request, 'url'|'method'>): HttpEffect<Response>;
export function post(url: string, request?: Omit<Request|RequestProgress, 'url'|'method'>): HttpEffect<unknown> {
  return send({ ...request, method: 'POST', url });
}


/**
 * Реализания запроса
 */
export function doXHR(req: Request, onNext: (x: Either<HttpError, Response>) => void, onComplete: () => void): () => void;
export function doXHR(req: RequestProgress, onNext: (x: Either<HttpError, Either<Progress, Response>>) => void, onComplete: () => void): () => void;
export function doXHR(req: Request|RequestProgress, onNext: (x: Either<HttpError, any>) => void, onComplete: () => void): () => void {
  const onSuccess = (x: Response) => ('progress' in req && req.progress ? onNext(either.right(either.right(x))) : onNext(either.right(x)), onComplete());
  const onProgress = (x: Progress) => 'progress' in req && req.progress ? onNext(either.right(either.left(x))) : void 0;
  const onFailure = (x: HttpError) => (onNext(either.left(x)), onComplete());
  
  const xhr = new XMLHttpRequest();
  xhr.addEventListener('error', () => onFailure({ tag: 'NetworkError' }));
  xhr.addEventListener('timeout', () => onFailure({ tag: 'Timeout' }));
  xhr.addEventListener('load', () => onSuccess({
    url: xhr.responseURL,
    status: xhr.status,
    statusText: xhr.statusText,
    headers: parseHeaders(xhr.getAllResponseHeaders()),
    body: xhr.response || xhr.responseText,
  }));
  try {
    xhr.open(req.method, req.url, true);
  } catch (e) {
    onFailure({ tag: 'BadUrl', desc: req.url });
  }

  if ('progress' in req && req.progress) {
    xhr.addEventListener('progress', e => onProgress(e.lengthComputable ? { tag: 'Computable', loaded: e.loaded, total: e.total } : { tag: 'Uncomputable' }));
  }
  if (req.timeout) xhr.timeout = req.timeout;
  if (typeof (req.withCredentials) !== 'undefined') xhr.withCredentials = req.withCredentials;
  if (typeof (req.headers) !== 'undefined') {
    for (let key in req.headers) {
      if (!req.headers.hasOwnProperty(key)) continue;
      const value = req.headers[key];
      if (typeof(value) !== 'undefined' && value !== null)
        xhr.setRequestHeader(key, String(value));
    }
  }
  const body = Object.prototype.toString.apply(req.body) === '[object Object]' ? JSON.stringify(req.body) : req.body;
  xhr.send(body);
  return () => xhr.abort();
}


/** parse response as JSON */
export function expectJSON<A>(decoder: Decoder<A>): (resp: Response) => Eff<HttpError, A> {
  return resp => {
    if (resp.body === '') return eff.failure<HttpError>({ tag: 'BadPayload', desc: 'empty body' });
    let val = null;
    try {
      val = JSON.parse(resp.body);
    } catch (e) {
      return eff.failure<HttpError>({ tag: 'BadPayload', desc: 'invalid json' });
    }
    return eff.fromEither(
      decoder.validate(val).mapLeft<HttpError>(problem => ({ tag: 'ValidationProblem', problem, url: resp.url }))
    );
  };
}


/** Parse headers from string to dict */
function parseHeaders(rawHeaders: string): Record<string, string> {
  const output = {};
  const lines = rawHeaders.split('\r\n');
  for (let i in lines) {
    const index = lines[i].indexOf(': ');
    if (index < 0) continue;
    const key = lines[i].substring(0, index);
    const value = lines[i].substring(index + 2);
    output[key] = value;
  }
  return output;
}


/** Build an url */
export function join(...args: Array<string|Params>): string {
  let path = '';
  let params = {} as Record<string, string>;
  let query = '';

  for (let i in args) {
    const arg = args[i];
    if (typeof (arg) === 'string') path = joinTwo(path, arg);
    else Object['assign'](params, arg);
  }

  for (let key in params) {
    if (!params.hasOwnProperty(key) || typeof(params[key]) === 'undefined' || params[key] === null) continue;
    if (Array.isArray(params[key])) {
      for (const v of params[key]) {
        if (typeof(params[key]) === 'undefined' || params[key] === null) continue;
        query += (query ? '&' : '') + `${encodeURIComponent(key)}=${encodeURIComponent(v)}`;
      }
    } else {
      query += (query ? '&' : '') + `${encodeURIComponent(key)}=${encodeURIComponent(params[key])}`;
    }
  }

  return query ? (path + '?' + query) : path;
  
  /** Join segments of url */
  function joinTwo(a: string, b: string): string {
    if (a === '') return b;
    if (b === '') return a;
    const trailing = a.length && a[a.length - 1] === '/';
    const leading = b.length && b[0] === '/';
    if (trailing && leading) return a.substring(0, a.length - 1) + b;
    if (!trailing && !leading) return a + '/' + b;
    return a + b;
  }
}


// Helper
export type Omit<T, K> = Pick<T, Exclude<keyof T, K>>;
