import * as React from 'react';
import memoize from '~/functions/memoize';
import { ObjectKey, ObjectPath } from '~/functions/get-at';
import { isEqual } from 'lodash';
import { AuthCtx as Ctx } from '~/context';
import { I18nString } from '~/gettext';


// Контекст поля
export interface FieldProps<T = any> {
  ctx?: Ctx;
  value: T;
  disabled?: Disabled;
  // Здесь должен передавться результат `validate` те `Validation<A>`
  error?: Error;
  onValueChange?(value: T): void;
  onValueChange?(value: any, at?: ObjectPath): void;
}


// Флаги активности
export type Disabled = boolean|object;


// Ошибки в поле
export type Error = boolean|string|object|I18nString;


// Валидация формы
export type Validation<A> = Partial<Record<keyof A, Error>>;


// Props
export type CardProps<T> = {
  ctx?: any,
  model: {
    initial: T;
    modified: T|null;
  };
  dispatch(action: { tag: 'Change', value, at }): void;
}


// Опции для `createForm`
export type CreateFormOptions<T> = {
  validate?(value: T): any;
  disabled?(value: T, instance): any;
};


export default function createForm<O extends CreateFormOptions<any>>(options: O) {
  type T = O extends CreateFormOptions<infer T> ? T : never;
  
  function bindZoom(self: React.Component<CardProps<T>>) {
    function Zoom<K1 extends keyof T, C extends Any<T[K1]>>(...keys: [K1]): FieldProps<T[K1]>;
    function Zoom<K1 extends keyof T, K2 extends keyof T[K1], C extends Any<T[K1][K2]>>(...keys: [K1, K2]): FieldProps<T[K1][K2]>;
    function Zoom<K1 extends keyof T, K2 extends keyof T[K1], K3 extends keyof T[K1][K2], C extends Any<T[K1][K2][K3]>>(...keys: [K1, K2, K3]): FieldProps<T[K1][K2][K3]>;
    function Zoom(...keys) {
      const { dispatch, ctx } = self.props;
      const value = self.props.model.modified || self.props.model.initial;
      const error = options.validate ? options.validate(value) : {};
      const disabled = options.disabled ? options.disabled(value, self) : {};
      const onValueChange = (value, at: ObjectPath=[]) => dispatch({ tag: 'Change', value, at });
      
      return {
        ctx,
        value: zoomAny(keys, value),
        error: zoomAny(keys, error),
        disabled: zoomAny(keys, disabled),
        onValueChange: zoomOnChange(keys, onValueChange),
      };
    }
    return Zoom;
  }

  function set<M extends CardProps<T>['model']>(model: M, form: T): M {
    // @ts-ignore
    return isEqual(model.initial, form) ? { ...model, modified: null } : { ...model, modified: form };
  }

  function get<M extends CardProps<T>['model']>(model: M): T {
    // @ts-ignore
    return model.modified || model.initial;
  }

  function modify<M extends CardProps<T>['model']>(model: M, f: (form: T) => T): M {
    return set(model, f(get(model)));
  }

  // @ts-ignore
  return { ...options, bindZoom, set, get, modify } as { bindZoom: typeof bindZoom, set: typeof set, get: typeof get, modify: typeof modify } & O;
}


const zoomOnChange = memoize((path: ObjectKey[], onChange: ((x: any, at: ObjectPath) => void)|undefined) => {
  return (next, at=[]) => {
    onChange && onChange(next, [...path, ...at]);
  };
});


const zoomAny = memoize((path: ObjectKey[], input: any) => {
  let iter = input as any;
  for (const key of path) {
    if (typeof(iter) !== 'object' || iter === null) return iter;
    iter = iter[key];
  }
  return iter;
});


// Хелперы
export type Any<T=any> = React.ReactType<FieldProps<T>>;
export type GetProps<T extends Any> = T extends React.ReactType<infer Props> ? Props : never;
