import { Either } from '~/either';
import { Expr, absurd } from '~/types';


// This library helps you declaratively construct type-safe validators for untyped JSON
// inspired by http://package.elm-lang.org/packages/elm-lang/core/3.0.0/Json-Decode
// and https://github.com/gcanti/io-ts


/** ADT */
export type Decoder<A> =
  | CustomDecoder<A>
  | ArrayDecoder<A>
  | DictionaryDecoder<A>
  | RecordDecoder<A>
  | AtDecoder<A>
  | PrimitiveDecoder<A>
  | PureDecoder<A>
  | ChainDecoder<A>
  | OneOfDecoder<A>
  | DiscriminateOnDecoder<A>


/**
 * Информация о возникшей ошибке
 */
export type Problem = {
  value: any;
  rootValue: any;
  decoders: Decoder<any>[];
  message: string;
  path: Array<string|number>;
}


/**
 * Опции для `DecoderBase.prototype.validate`
 */
export interface ValidateOptions {
  printProblem: boolean;
}

const defaultValidateOptions: ValidateOptions = {
  printProblem: false,
};


// Базовый класс для наследования методов
export class DecoderBase<A> {
  readonly _A: A;
  private static path: Decoder<any>[] = [];
  private static rootValue: any = undefined;

  /**
   * Печать декодера в виде выражения которым он был создан
   */
  prettyPrint(): string {
    const self = this as any as Decoder<A>;
  }

  /**
   * Валидация произвольного значения
   */
  validate(value: any, options?: Partial<ValidateOptions>): Either<Problem, A> {
    const self = this as any as Decoder<A>;
    let _options: ValidateOptions;
    
    let cleanUpRoot = false;
    DecoderBase.path.push(self);
    if (DecoderBase.rootValue === undefined) { DecoderBase.rootValue = value; cleanUpRoot = true; }
    const output = doValidate(self, value)
    DecoderBase.path.pop();
    if (cleanUpRoot) DecoderBase.rootValue = undefined;

    // Печать проблемы в консоль если выставлена соответствующая опция
    if (output.tag === 'Left' && makeOptions().printProblem) {
      printProblems(output.value);
    }
    return output;

    function makeOptions(): ValidateOptions {
      if (_options) return _options;
      return _options = options ? { ...defaultValidateOptions, ...options } : defaultValidateOptions;
    }

    function doValidate(decoder: Decoder<A>, value: any): Either<Problem, A> {
      switch(decoder.tag) {
	case 'CustomDecoder': return decoder.validateCustom(value).mapLeft(decorateProblem);;
	case 'ArrayDecoder': return decoder.validateArray(value).mapLeft(decorateProblem);;
	case 'DictionaryDecoder': return decoder.validateDictionary(value).mapLeft(decorateProblem);;
	case 'RecordDecoder': return decoder.validateRecord(value).mapLeft(decorateProblem);;
	case 'AtDecoder': return decoder.validateAt(value).mapLeft(decorateProblem);;
	case 'PrimitiveDecoder': return decoder.validatePrimitive(value).mapLeft(decorateProblem);;
	case 'PureDecoder': return decoder.value.mapLeft(decorateProblem);;
	case 'OneOfDecoder': return decoder.validateOneOf(value).mapLeft(decorateProblem);;
	case 'ChainDecoder': return decoder.andThen(decoder.decoder.validate(value)).validate(value);;
      }
    }
    
    function decorateProblem(problem: Problem|string): Problem {
      return typeof(problem) === 'string'
	? { value, rootValue: DecoderBase.rootValue, decoders: DecoderBase.path.slice(), message: problem, path: [] }
	: problem;
    }
  }

  map<B>(f: (a: A) => B): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, x => of(f(x))));
  }

  mapTo<B>(value: B): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, () => of(value)));
  }

  chain<B>(f: (a: A) => Decoder<B>): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, f));
  }
  
  chainTo<B>(value: Decoder<B>): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, () => value));
  }
  
  mapProblem(f: (a: Problem) => Problem): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(f, x => x));
  }
  
  mapProblemTo(value: Problem): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(() => value, x => x));
  }

  withDefault(defValue: A): Decoder<A>;
  withDefault<B extends Expr>(defValue: B): Decoder<A|B>;
  withDefault<B extends Expr>(defValue: B): Decoder<A|B> {
    return new OneOfDecoder([this as any, of(defValue)]);
  }

  refine(pred: (a: A) => boolean): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, ethr => {
      switch (ethr.tag) {
        case 'Left': return new PureDecoder(ethr);
        case 'Right': return pred(ethr.value) ? new PureDecoder(ethr) : fail('refinement failed');
      }
    });
  }
}




/**
 * Декодер с произвольной функцией для валидации
 */
export class CustomDecoder<A> extends DecoderBase<A> {
  readonly tag: 'CustomDecoder' = 'CustomDecoder';

  constructor(
    readonly name: string,
    readonly validateCustom: (val: any) => Either<Problem|string, A>,
  ) { super(); }
}


/** 
 * Декодер массивов
 */
export class ArrayDecoder<A> extends DecoderBase<A> {
  readonly tag: 'ArrayDecoder' = 'ArrayDecoder';

  constructor(
    readonly decoder: Decoder<any>,
  ) { super(); }

  validateArray(value: any): Either<Problem|string, A> {
    const output: any[] = [];
    if (!Array.isArray(value)) return Either.failure('not an array');
    for (let i = 0; i < value.length; i++) {
      const ethr = this.decoder.validate(value[i]);
      switch(ethr.tag) {
        case 'Left': { ethr.value.path.push(i); return ethr; }
        case 'Right': output.push(ethr.value); break;
      }
    }
    return Either.of(output as any as A);
  }
}


/** 
 * Декодер словарей (или хешей, те объектов с произвольным кол-вом ключей)
 */
export class DictionaryDecoder<A> extends DecoderBase<A> {
  readonly tag: 'DictionaryDecoder' = 'DictionaryDecoder';

  constructor(
    readonly decoder: Decoder<any>,
  ) { super(); }

  validateDictionary(value: any): Either<Problem|string, A> {
    if (value === null) return Either.failure('found null');
    if (typeof (value) !== 'object') return Either.failure('not an object');
    const output: { [k: string]: A } = {};
    for (let key in value) {
      if (!value.hasOwnProperty(key)) continue;
      const ethr = this.decoder.validate(value[key]);
      switch(ethr.tag) {
        case 'Left': { ethr.value.path.push(key); return ethr; }
        case 'Right': output[key] = ethr.value; break;
      }
    }
    return Either.of(output as any as A);
  }
}


/** 
 * Декодер записей (объекты с фиксированным количеством полей)
 */
export class RecordDecoder<A> extends DecoderBase<A> {
  readonly tag: 'RecordDecoder' = 'RecordDecoder';

  constructor(
    readonly description: Record<string, Decoder<any>>,
  ) { super(); }

  validateRecord(value: any): Either<Problem|string, A> {
    if (value === null) return Either.failure('found null');
    if (typeof (value) !== 'object') return Either.failure('not an object');
    const output: { [k: string]: any } = {};
    for (let key in this.description) {
      if (!this.description.hasOwnProperty(key)) continue;
      const ethr = this.description[key].validate(value[key]);
      switch(ethr.tag) {
        case 'Left': { ethr.value.path.push(key); return ethr; }
        case 'Right': output[key] = ethr.value; break;
      }
    }
    return Either.of(output as any as A);
  }
}


/**
 * Декодер поля с указанным индексом
 */
export class AtDecoder<A> extends DecoderBase<A> {
  readonly tag: 'AtDecoder' = 'AtDecoder';

  constructor(
    readonly path: Array<string|number>,
    readonly decoder: Decoder<A>,
  ) { super(); }


  validateAt(value: any): Either<Problem|string, A> {
    let iter = value;
    for (let i in this.path) {
      if (iter === undefined || !iter.hasOwnProperty(this.path[i])) {
        iter = undefined;
        break;
      }
      iter = iter[this.path[i]];
    }
    return this.decoder.validate(iter);
  }
}


/**
 * Декодер для примитовов
 */
export class PrimitiveDecoder<A> extends DecoderBase<A> {
  readonly tag: 'PrimitiveDecoder' = 'PrimitiveDecoder';

  constructor(
    readonly primitive: 'null'|'undefined'|'string'|'boolean'|'any'|'nat'|'int'|'float'
  ) { super(); }

  validatePrimitive(value: any): Either<string, A> {
    switch (this.primitive) {
      case 'null': return value === null ? Either.of(value) : Either.failure(`expected null, got ${fancyTypeOf(value)}`);
      case 'undefined': return value === undefined ? Either.of(value) : Either.failure(`expected undefined, got ${fancyTypeOf(value)}`);
      case 'string': return typeof(value) === 'string' ? Either.of(value as any) : Either.failure(`expected a string, got ${fancyTypeOf(value)}`);
      case 'boolean': return typeof(value) === 'boolean' ? Either.of(value as any) : Either.failure(`expected a boolean, got ${fancyTypeOf(value)}`);
      case 'any': return Either.of(value);
      case 'nat': return typeof (value) !== 'number' ? Either.failure('not a number') : (value|0) === value && value >= 0 ? Either.of(value as any) : Either.failure('not a natural number');
      case 'int': return typeof (value) !== 'number' ? Either.failure('not a number') : (value|0) === value ? Either.of(value as any) : Either.failure('not an integer')
      case 'float': return typeof (value) !== 'number' ? Either.failure('not a number') : Either.of(value as any);
    }
  }
}


/** 
 * Тривиальный декодер
 */
export class PureDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly value: Either<Problem|string, A>,
  ) { super(); }
}


/**
 * Монадный комбинатор
 */
export class ChainDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly decoder: Decoder<any>,
    readonly andThen: (x: Validation<any>) => Decoder<A>,
  ) { super(); }
}


/** 
 * `oneOf` комбинатор
 */
export class OneOfDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly alternatives: Decoder<any>[],
  ) { super(); }

  validateOneOf(value: any): Either<Problem|string, A> {
    for (const decoder of this.alternatives) {
      const ethr = decoder.validate(value);
      switch(ethr.tag) {
        case 'Left': break;
        case 'Right': return ethr;
      }
    }
    return Either.failure('none of decoders succeded');
  }
}


/** 
 * `discriminateOn` комбинатор
 */
export class DiscriminateOnDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly discriminator: string|number,
    readonly alternatives: Record<string|number, Decoder<unknown>>,
  ) { super(); }
}


/** 
 * `discriminateOn` комбинатор
 */
export class WithDefaultDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly decoder: Decoder<A>,
    readonly defaultValue: A,
  ) { super(); }
}


/** 
 * Тип результата для функции-валидатора
 */
export type Validation<A> = Either<Problem|string, A>;


/**
 * Конструктир кастомных декодеров
 */
export function decoder<A>(validate: (value: any) => Either<Problem|string, A>): CustomDecoder<A>;
export function decoder<A>(name: string, validate: (value: any) => Either<Problem|string, A>): CustomDecoder<A>;
export function decoder(): any {
  if (arguments.length === 1) return new CustomDecoder('custom', arguments[0]);
  if (arguments.length === 2) return new CustomDecoder(arguments[0], arguments[1]);
  throw new TypeError(`decoder: invalid number of arguments`);
}


/**
 * Алиас для `x => new PureDecoder(Either.of(x))`
 */
export function of<A extends Expr>(a: A): PureDecoder<A> {
  return new PureDecoder(Either.of(a));
}


/**
 * Алиас для `x => new PureDecoder(Either.failure(x))`
 */
export function fail(x: Problem|string): Decoder<never> {
  return new PureDecoder(Either.failure(x));
}


/** 
 * Аппликативный комбинатор
 * TODO: сделать отдельным вариантом в ADT для реализации `prettyPrint`
 */
export function ap<A,B>(a: Decoder<A>, f:(a: A) => B): CustomDecoder<B>;
export function ap<A,B,C>(a: Decoder<A>, b: Decoder<B>, f:(a: A, b: B) => C):CustomDecoder<C>;
export function ap<A,B,C,D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, f:(a: A, b: B, c: C) => D):CustomDecoder<D>;
export function ap<A,B,C,D,E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, f:(a: A, b: B, c: C, d: D) => E):CustomDecoder<E>;
export function ap<A,B,C,D,E,F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f:(a: A, b: B, c: C, d: D, e: E) => F):CustomDecoder<F>;
export function ap<A,B,C,D,E,F,G>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, f:(a: A, b: B, c: C, d: D, e: E, f: F) => G):CustomDecoder<G>;
export function ap<A,B,C,D,E,F,G,H>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G) => H):CustomDecoder<H>;
export function ap<A,B,C,D,E,F,G,H,J>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H) => J):CustomDecoder<J>;
export function ap<A,B,C,D,E,F,G,H,J,K>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, j: Decoder<J>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H, j: J) => K):CustomDecoder<K>;
export function ap<A,B,C,D,E,F,G,H,J,K,L>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, j: Decoder<J>, k: Decoder<K>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H, j: J, k: K) => L):CustomDecoder<L>;
export function ap(...args: Array<Decoder<any> | Function>): Decoder<any> {
  return decoder('ap', (val) => {
    const func = args[args.length - 1] as Function;
    const results: Array<any> = [];
    for (let i = 0; i < args.length - 1; i++) {
      const dec = args[i] as Decoder<any>;
      const ethr = dec.validate(val);
      switch(ethr.tag) {
        case 'Left': return ethr;
        case 'Right': results.push(ethr.value); break;
      }
    }
    return Either.of(func.apply(undefined, results));
  });
}


// Примитивы
const anyDecoder = new PrimitiveDecoder<any>('any');
const stringDecoder = new PrimitiveDecoder<string>('string');
const booleanDecoder = new PrimitiveDecoder<boolean>('boolean');
const nullDecoder = new PrimitiveDecoder<null>('null');
const undefinedDecoder = new PrimitiveDecoder<undefined>('undefined');
export const nat = new PrimitiveDecoder<number>('nat');
export const int = new PrimitiveDecoder<number>('int');
export const float = new PrimitiveDecoder<number>('float');


// Экспорт с переименованием
export { anyDecoder as any, stringDecoder as string, booleanDecoder as boolean, nullDecoder as null, undefinedDecoder as undefined };


/**
 * Сопоставление с несколькими декодерами до первого успешного
 * сравнвния
 */
export function oneOf<A>(a: Decoder<A>): OneOfDecoder<A>;
export function oneOf<A,B>(a: Decoder<A>, b: Decoder<B>): OneOfDecoder<A|B>;
export function oneOf<A,B,C>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>): OneOfDecoder<A|B|C>;
export function oneOf<A,B,C,D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>): OneOfDecoder<A|B|C|D>;
export function oneOf<A,B,C,D,E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>): OneOfDecoder<A|B|C|D|E>;
export function oneOf<A,B,C,D,E,F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>): OneOfDecoder<A|B|C|D|E|F>;
export function oneOf<array extends Decoder<any>[]>(array: array): Decoder<array[number]['_A']>;
export function oneOf(): Decoder<any> {
  const decoders = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  return new OneOfDecoder(decoders);
}


export function array<A>(decoder: Decoder<A>): Decoder<A[]> {
  return new ArrayDecoder(decoder);
}

export function record<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}

// Вложенные записи в декодерах
export function record2<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}

// Вложенные записи в декодерах
export function record3<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}


export const d = array(record({ a: array(record({ b: record({ c: stringDecoder }) })) } ));

export function dict<A>(decoder: Decoder<A>): Decoder<Record<string, A>> {
  return new DictionaryDecoder(decoder);
}

export function at<A>(path: string|string[]|number|number[], decoder: Decoder<A>): Decoder<A> {
  return new AtDecoder(Array.isArray(path) ? path : [path], decoder);
}

export const date: Decoder<Date> = decoder('date', (value) => {
  if (typeof (value) !== 'string') return Either.failure('not a string');
  const d = new Date(value);
  return isNaN(d.getTime()) ? Either.failure('error parsing date from string') : Either.of(d);
});


/** 
 * Хелпер для объявления декодеров в стиле Elm. Используется вместе в
 * `ap`
 */
export function required<A>(key: string|string[], dec: Decoder<A>): Decoder<A> {
  return at(key, dec);
}


/** 
 * Хелпер для объявления декодеров в стиле Elm. Используется вместе в
 * `ap`
 */
export function optional<A>(key: string|string[], dec: Decoder<A>, def: A): Decoder<A>;
export function optional<A,B>(key: string|string[], dec: Decoder<A>, def: B): Decoder<A|B>;
export function optional(key: string|string[], dec: Decoder<any>, def: any): Decoder<any> {
  return required(key, dec).withDefault(def);
}


/** 
 * Создание декодера перечислением всех допустимых значений 
 */
export type L = string|number|boolean|null;
export function literals<A extends L>(a: A): Decoder<A>;
export function literals<A extends L, B extends L>(a: A, b: B): Decoder<A|B>;
export function literals<A extends L, B extends L, C extends L>(a: A, b: B, c: C): Decoder<A|B|C>;
export function literals<A extends L, B extends L, C extends L, D extends L>(a: A, b: B, c: C, d: D): Decoder<A|B|C|D>;
export function literals<A extends L, B extends L, C extends L, D extends L, E extends L>(a: A, b: B, c: C, d: D, e: E): Decoder<A|B|C|D|E>;
export function literals<A extends L, B extends L, C extends L, D extends L, E extends L, F extends L>(a: A, b: B, c: C, d: D, e: E, f: F): Decoder<A|B|C|D|E|F>;
export function literals<A extends L, B extends L, C extends L, D extends L, E extends L, F extends L, G extends L>(a: A, b: B, c: C, d: D, e: E, f: F, g: G): Decoder<A|B|C|D|E|F|G>;
export function literals<A extends L, B extends L, C extends L, D extends L, E extends L, F extends L, G extends L, H extends L>(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H): Decoder<A|B|C|D|E|F|G|H>;
export function literals<A extends L[]>(array: A): Decoder<A[number]>;
export function literals(): Decoder<any> {
  const literals: Expr[] = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  return new OneOfDecoder(literals.map(x => decoder(v => v === x ? Either.of(v) : Either.failure(`expected ${x}, got ${fancyTypeOf(v)}`))));
}


/**
 * Кортежи разных размеров. Проверяемое значение необязательно должно
 * быть массивом
 * ```ts
 *   const pair = t.tuple(t.string, t.number);
 *   const pair_2 = t.record({ '0': t.string, '1': t.number }); // тоже самое
 * ```
 */
export function tuple<A>(a: Decoder<A>): Decoder<[A]>;
export function tuple<A, B>(a: Decoder<A>, b: Decoder<B>): Decoder<[A, B]>;
export function tuple<A, B, C>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>): Decoder<[A, B, C]>;
export function tuple<A, B, C, D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>): Decoder<[A, B, C, D]>;
export function tuple<A, B, C, D, E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>): Decoder<[A, B, C, D, E]>;
export function tuple<A, B, C, D, E, F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>): Decoder<[A, B, C, D, E, F]>;
export function tuple<A, B, C, D, E, F, G>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>): Decoder<[A, B, C, D, E, F, G]>;
export function tuple<A, B, C, D, E, F, G, H>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>): Decoder<[A, B, C, D, E, F, G, H]>;
export function tuple<A, B, C, D, E, F, G, H, I>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>, i: Decoder<I>): Decoder<[A, B, C, D, E, F, G, H, I]>;
export function tuple<A, B, C, D, E, F, G, H, I, J>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>, i: Decoder<I>, j: Decoder<J>): Decoder<[A, B, C, D, E, F, G, H, I, J]>;
export function tuple<A>(args: Decoder<A>[]): Decoder<A[]>;
export function tuple(): Decoder<any> {
  const args: Decoder<any>[] = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  // @ts-ignore
  return ap(...args.map((decoder, idx) => at(idx, decoder)), (...xs) => xs);
}


/** 
 * Печать проблемы с консоль
 */
export function printProblems(problems: Problem): void {
  const lastDecoder = problems.decoders[problems.decoders.length - 1];
  console.log(`%cValidation error: ${problems.message}`, 'color: #f00; font-size: 16px; font-weight: 600;')
  console.log('%cfailed decoder:', 'font-weight: 600;', lastDecoder.prettyPrint());
  console.log('%cfailed value:', 'font-weight: 600;', problems.value);
  console.log('%croot value:', 'font-weight: 600;', problems.rootValue);
  console.log('%cproblem path:', 'font-weight: 600;', prettyPrintPath(problems.path));
  //console.log('decoders ', problems.decoders.map(x => x.prettyPrint()));
}


// Хелпер
function fancyTypeOf(value: any): string {
  return Object.prototype.toString.call(value);
}


// Хелпер
function prettyPrintPath(path: Array<string|number>): string {
  let output = '_';
  for (let i = path.length - 1; i >= 0; i--) {
    if (/^[a-zA-Z_$][\w_$]*$/.test('' + path[i])) output += '.' + path[i];
    else output += `[${JSON.stringify(path[i])}]`;
  }
  return output;
}


/**
 * Печать декодера.
 * ```ts
 * const user = t.record({ name: t.string });
 * console.log(t.prettyPrint(user)); // => "t.record({ name: t.string })"
 * ```
 */
export function prettyPrint(decoder: Decoder<any>): string {
  if (decoder instanceof CustomDecoder) return `t.decoder(${JSON.stringify(decoder.name)}, <func>)`;
  if (decoder instanceof ArrayDecoder) return `t.array(${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof DictionaryDecoder) return `t.dict(${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof RecordDecoder) return `t.record({ ${Object.keys(decoder.description).map(k => decoder.description[k].prettyPrint()).join(', ')} })`;
  if (decoder instanceof AtDecoder) return `t.at(${JSON.stringify(decoder.path)}, ${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof PrimitiveDecoder) return `t.${decoder.primitive}`;
  if (decoder instanceof PureDecoder) return `t.of(${JSON.stringify(decoder.value)})`;
  if (decoder instanceof ChainDecoder) return `${decoder.prettyPrint()}.chain(<func>)`;
  if (decoder instanceof OneOfDecoder) return `t.oneOf(${decoder.alternatives.map(x => x.prettyPrint()).join(', ')})`;
  if (decoder instanceof DiscriminateOnDecoder) {
    const discriminator = JSON.stringify(decoder.discriminator);
    const alternatives = Object.keys(decoder.alternatives).map(k => JSON.stringify(k) + ": " + decoder.alternatives[k].prettyPrint()).join(', ');
    return `t.discriminateOn(${}, ${alternatives})`;
  }
  return absurd(decoder);
}
