import * as React from 'react';
import * as ReactDOM from 'react-dom';
import Suggestions, { Props as SuggestionProps } from '~/fields/Suggestions';
import { AuthCtx as Ctx } from '~/context';
import { FieldProps } from '~/create-form';
import TextField from '~/fields/TextField';
import { Props as TextFieldProps } from '~/fields/TextField';
import { StandardProps, ListItem } from '@material-ui/core';
import Icon from './Icon';
import withStyles, { StyleRules } from '@material-ui/core/styles/withStyles';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import { Source, State as AutoCompleteState } from '~/fields/AutoComplete/state-machine';
import * as ST from '~/fields/AutoComplete/state-machine';
import * as Rx from 'rxjs';
import memoize from '~/functions/memoize';
import { pick } from 'lodash';
import classNames = require('classnames');
import { withGettext, Gettext } from '~/gettext';
import { fade } from '@material-ui/core/styles/colorManipulator';


// Props
export type Props<A> = StandardProps<React.HTMLProps<HTMLDivElement>, string, 'value'|'disabled'> & FieldProps<A> & {
  ctx?: Ctx;
  source: Source<A>;
  debounce?: number;
  renderItem?: SuggestionProps['renderSuggestion'];
  suggestionProps?: Partial<SuggestionProps>;
  printItem?(value: A): string;
  keepOpenAfterSelect?: boolean;
  textFieldProps?: Partial<TextFieldProps>;
  openOnFocus?: boolean; // true по умолчанию
  openOnClick?: boolean; // true по умолчанию
  anchorEl?: HTMLElement;
  nonNull?: boolean;
  fullWidth?: boolean;
  placeholder?: string;
  null?: A;
  observable?: Rx.Observable<ST.Action<A>>;
  children?: React.ReactElement<React.HTMLProps<HTMLInputElement>>;
}


// State
export type State<A> = AutoCompleteState<A> & {
  value: string|null;
}


// Component
// @ts-ignore Хак для работы дженерик-параметров
@withStyles(styles)
// @ts-ignore
@withGettext(require('./i18n'))
export default class AutoComplete<A=string> extends React.Component<Props<A> & { __: Gettext }, State<A>> {
  static defaultProps = {
    openOnFocus: true,
    openOnClick: true,
    fullWidth: true,
    printItem: x => x ? String(x) : '',
  } as any;
  
  state: State<A> = { ...ST.init(), value: null };
  anchorEl: HTMLElement|null; // Ссылка на <input/>
  rectEl: HTMLElement|null; // Ссылка на элемент для выравнивания, может быть тем же что и `inputEl`
  debounceTimer: number|null = null;
  subscription: Rx.Subscription|null = null;
  paperPrevScroll: number|null = null;
  paperEl: HTMLDivElement|null = null;

  // Выполнение действий из ST
  dispatch = (action: ST.Action<A>) => {
    const { ctx, openOnFocus, openOnClick, source } = this.props;
    const acCtx = { ...ctx, options: { openOnFocus, openOnClick, source } };
    this.setState(prev => {
      const [next, command] = ST.update(acCtx, action, prev);
      command.subscribe(ethr => ethr.tag === 'Right' && this.dispatch(ethr.value));
      return next;
    });
  };

  handleValueChange = (value: string) => {
    const debounce = this.props.debounce || 500;
    this.setState({ value });
    if (this.debounceTimer) clearTimeout(this.debounceTimer);
    if (!value) {
      !this.props.nonNull && this.handleSuggestionSelect('null' in this.props ? this.props.null : null);
    }
    this.debounceTimer = setTimeout(() => (this.debounceTimer = null, this.dispatch({ tag: 'Search', value })), debounce) as any;
  };
  
  handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    this.handleValueChange(e.target.value);
  };

  handleFocus = (e?: React.FocusEvent<any>) => {
    this.dispatch({ tag: 'Focus' })
    this.props.onFocus && e && this.props.onFocus(e);
  };

  handleBlur = (e?: React.FocusEvent<any>) => {
    this.dispatch({ tag: 'Blur' });
    this.setState({ value: null });
    this.props.onBlur && e && this.props.onBlur(e);
  };

  handleClick = (e?: React.SyntheticEvent) => {
    this.dispatch({ tag: 'Click' });
  };

  handleToggleVisibility = () => {
    this.state.open ? this.dispatch({ tag: 'Close' }) : this.dispatch({ tag: 'Open' });
  };

  handleKeyDown = (e: React.KeyboardEvent) => {
    switch (e.key) {
      case 'ArrowDown': {
        !this.state.open && this.dispatch({ tag: 'Open' });
        break;
      }
    }
  };

  handleSuggestionSelect = (value: any) => {
    const { keepOpenAfterSelect, onValueChange, disabled } = this.props;
    this.setState({ value: null });
    !keepOpenAfterSelect && this.dispatch({ tag: 'Blur' });
    !disabled && onValueChange && onValueChange(value);
  };

  handleVisibilityChnage = (open: boolean) => {
    if (!open) this.dispatch({ tag: 'Close' });
    else this.dispatch({ tag: 'Open' });
  };

  handleCloseClick = (e: React.MouseEvent) => {
    e.stopPropagation();
    this.setState({ value: null });
    !this.props.nonNull && this.handleSuggestionSelect('null' in this.props ? this.props.null : null);
  };

  getValue = () => {
    const { printItem } = this.props;
    if (this.state.value !== null) return this.state.value;
    const { value } = this.props;
    return (printItem || String)(value!) || '';
  };

  componentDidMount() {
    this.anchorEl = ReactDOM.findDOMNode(this) as HTMLElement;;
  }

  componentDidUpdate() {
    if (this.paperEl && this.paperPrevScroll && this.paperEl.scrollTop !== this.paperPrevScroll) {
      this.paperEl.scrollTo(0, this.paperPrevScroll);
      this.paperEl = null;
      this.paperPrevScroll = null;
    }
  }

  componentWillReceiveProps(nextProps: Props<A>) {
    if (nextProps.observable !== this.props.observable) this.listen(nextProps);
  }

  componentWillUnmount() {
    this.unlisten();
    this.debounceTimer && (clearTimeout(this.debounceTimer), this.debounceTimer = null);
  }

  listen(props: Props<A>) {
    this.subscription && (this.subscription.unsubscribe(), this.subscription = null);
    if (!props.observable) return;
    this.subscription = props.observable.subscribe(this.dispatch);
  }

  unlisten() {
    this.subscription && (this.subscription.unsubscribe(), this.subscription = null);
  }

  endAdornment = memoize((nonNull: boolean, disabled: boolean, __: Gettext) => <React.Fragment>
    <Icon onClick={disabled ? undefined : this.handleToggleVisibility}>arrow_drop_down</Icon>
    {!nonNull && <Icon onClick={disabled ? undefined : this.handleCloseClick} data-rh={__('Clear')}>close</Icon>}
  </React.Fragment>);

  childrenProps = () => {
    const { nonNull, disabled, __ } = this.props;
    
    return {
      ...pick(this.props, 'disabled', 'error', 'fullWidth', 'placeholder', 'ctx'),
      value: this.getValue(),
      onValueChange: disabled ? undefined : this.handleValueChange,
      onChange: disabled ? undefined : this.handleChange,
      onFocus: disabled ? undefined : this.handleFocus,
      onBlur: disabled ? undefined : this.handleBlur,
      onKeyDown: disabled ? undefined : this.handleKeyDown,
      onClick: disabled ? undefined : this.handleClick,
      endAdornment: this.endAdornment(!!nonNull, !!disabled, __),
    };
  };

  handleSeeMore = (e: React.MouseEvent<HTMLAnchorElement>) => {
    e.preventDefault();
    e.stopPropagation();
    //    const { query: { offset, limit } } = this.state;
    this.paperEl = e.currentTarget.parentElement!.parentElement! as HTMLDivElement;
    this.paperPrevScroll = this.paperEl.scrollTop;
    this.dispatch({ tag: 'More' });
  };

  renderSeeMoreLink() {
    const { __, classes } = this.props;
    const { total, query: { limit, offset } } = this.state;
    if (total !== 0 && limit + offset >= total) return null;
    
    return <ListItem className={classes!.seeMore}>
      <a href="javascript://void 0" onMouseDown={this.handleSeeMore}>{__('See more')}</a>
    </ListItem>;
  }
  render() {
    const { className, fullWidth, children, ctx, source, debounce, renderItem, printItem, openOnFocus, openOnClick, suggestionProps, keepOpenAfterSelect, textFieldProps, classes, anchorEl, observable, ...rest } = this.props;
    const { suggestions, open, pending } = this.state;
    const rootClass = classNames(classes!.root, className, {
      [classes!.fullWidth!]: fullWidth
    });

    return <div {...rest} className={rootClass}>
      {!observable && (children ? React.cloneElement(children, this.childrenProps()) : <TextField {...this.childrenProps() as any} {...textFieldProps}/>)}
      <Suggestions
        {...suggestionProps}
        pending={pending}
        ctx={ctx!}
        anchorEl={anchorEl || this.anchorEl || undefined}
        open={open}
        suggestions={suggestions}
        renderSuggestion={renderItem || printItem}
        onSelect={this.handleSuggestionSelect}
        onVisibilityChange={this.handleVisibilityChnage}
        after={this.renderSeeMoreLink()}
      />
    </div>;
  }
}


// Props
export type CreateObservableProps = {
  children: React.ReactElement<TextFieldProps>;
  debounce?: number;
};


// Component
export class CreateObservable extends React.Component<CreateObservableProps> {
  observable = new Rx.Subject<ST.Action<any>>();
  debounceTimer: number|null = null;
  
  handleValueChange = (value: string) => {
    const { onValueChange } = this.props.children.props;
    const debounce = this.props.debounce || 500;
    if (this.debounceTimer) clearTimeout(this.debounceTimer);
    this.debounceTimer = setTimeout(() => (this.debounceTimer = null, this.observable.next({ tag: 'Search', value })), debounce) as any;
    onValueChange && onValueChange(value);
  };

  handleFocus: TextFieldProps['onFocus'] = (e) => {
    const { onFocus } = this.props.children.props;
    this.observable.next({ tag: 'Focus' });
    onFocus && onFocus(e);
  };

  handleBlur: TextFieldProps['onBlur'] = (e) => {
    const { onBlur } = this.props.children.props;
    this.observable.next({ tag: 'Blur' });
    onBlur && onBlur(e);
  };

  handleKeyDown: TextFieldProps['onKeyDown'] = (e) => {
    const { onKeyDown } = this.props.children.props;
    switch (e.key) {
      case 'ArrowDown': {
        this.observable.next({ tag: 'Open' });
        break;
      }
    }
    onKeyDown && onKeyDown(e);
  };

  componentWillUnmount() {
    this.debounceTimer && (clearTimeout(this.debounceTimer), this.debounceTimer = null);
  }
  
  render() {
    return React.cloneElement(this.props.children, {
      onValueChange: this.handleValueChange,
      onFocus: this.handleFocus,
      onBlur: this.handleBlur,
      onKeyDown: this.handleKeyDown,
    });
  }
}


// Style
export function styles(theme: Theme): StyleRules {
  const { unit } = theme.spacing;
  const linkStyles = {
    color: theme.palette.primary.main,
    textDecoration: 'none',
    '&:hover': {
      color: fade(theme.palette.primary.main, 0.75),
    },
    '&:active': {
      color: theme.palette.error.main,
    },
  };
  
  return {
    root: { 
      '& > div': {
        position: 'relative',
      }
    },

    fullWidth: {
      width: '100%',
    },

    seeMore: {
      marginBottom: -unit,
      '& a': {
        display: 'block',
        width: '100%',
        height: '100%',
        textAlign: 'center',
        ...linkStyles,
      },
    },
  };
}
