import * as React from 'react';


// Props
export type Props<A> = {
  delay?: number,
  children: React.ReactElement<{ value: A, onValueChange?(x: A): void, onBlur?(e: React.FocusEvent): void; }>;
}


// State
export type State<A> = {
  value?: A;
}


// Component
export class Debounce<A> extends React.Component<Props<A>, State<A>> {
  static defaultProps = {
    delay: 500,
  };
  state: State<A> = {};
  timer: number|null = null;
  callback: Function|null = null;

  componentWillReceiveProps(nextProps: Props<A>) {
    if ('value' in this.state) this.setState(prev => ({ value: undefined }));
  }

  handleChange = (value) => {
    if (this.timer) clearTimeout(this.timer);
    this.setState({ value })
    this.forceUpdate();
    this.callback = () => {
      this.timer = null;
      this.callback = null;
      const { onValueChange } = this.props.children.props;
      onValueChange && onValueChange(value);
    };
    this.timer = setTimeout(this.callback, this.props.delay);
  };
  
  handleBlur = (e: React.FocusEvent) => {
    this.callback && this.callback();
    const { onBlur } = this.props.children.props;
    onBlur && onBlur(e);
  };
  
  render() {
    return React.cloneElement(this.props.children, {
      value: this.state.value !== undefined ? this.state.value : this.props.children.props.value,
      onValueChange: this.handleChange,
      onBlur: this.handleBlur,
    });
  }
}

export default Debounce;
