import * as t from '../decoder';
import camelCase from 'lodash/camelCase';
import { absurd } from '../types';
import { WithDefault, RecordDecoder, Decoder } from '../decoder';
import { Either } from '../either';


/** 
 * Источник данных для чтения конфигов
 */
export type ConfigSource =
  | { tag: 'Cli', args: string[], prefix: string }
  | { tag: 'Location/search', value: string, transformKey(x: string): string }
  | { tag: 'JSON', value: object }
  | { tag: 'Concat', sources: ConfigSource[] }


export function cli(args: string[], prefix: string = ''): ConfigSource {
  return { tag: 'Cli', args, prefix };
}


export function queryString(value: string, transformKey = camelCase): ConfigSource {
  return { tag: 'Location/search', value, transformKey };
}


export function json(value: object): ConfigSource {
  return { tag: 'JSON', value };
}


export function concat(...sources: ConfigSource[]): ConfigSource {
  return { tag: 'Concat', sources };
}


export function validate<A>(config: ConfigSource, decoder: RecordDecoder<A>) {
  return decoder.validate(merge(decoder, config));
}


export function merge<A>(decoder: RecordDecoder<A>, ...srcs: ConfigSource[]): object {
  const source = srcs.length === 1 ? srcs[0] : concat(...srcs);
  if (source.tag === 'Cli') {
    const value: Record<string, any> = {};
    for (let i = 0; i < source.args.length; i++) {
      const prefixLen = '--'.length + source.prefix.length;
      const rest = source.args[i].substr(prefixLen);
      const [k, v] = rest.indexOf('=') === -1 ? [rest, source.args[++i] || ''] : source.args[i].split('=');
      // Проверка того что это известно
      if (!decoder._description.hasOwnProperty(k)) continue;
      value[k] = fromString(decoder._description[k], v);
    }
    return value;    
  }

  if (source.tag === 'Location/search') {
    const substring = source.value[0] === '?' ? source.value.substr(1) : source.value;
    const value = substring.split('&').map(x => x.split('=')).reduce((acc, [k, v]: any) => {
      const key = source.transformKey(decodeURIComponent(k));
      if (!decoder._description.hasOwnProperty(key)) return acc;
      acc[key] = fromString(decoder._description[key], decodeURIComponent(v));
      return acc;
    }, {});
    return value;
  }

  if (source.tag === 'JSON') {
    return source.value;
  }

  if (source.tag === 'Concat' ) {
    // @ts-ignore
    return Object.assign({}, ...source.sources.map(x => merge(decoder, x)));
  }
  
  return absurd(source);
}


export function fromString(d: Decoder<any>, value: string): unknown {
  if (d instanceof t.ArrayDecoder) {
    return value.split(', ').map(x => fromString(d._decoder, x));
  }

  if (d instanceof t.Dict) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }
  
  if (d instanceof t.RecordDecoder) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }

  if (d instanceof t.Primitive) {
    if (d._type === 'string') return value;
    if (d._type === 'boolean') return !(value === 'false' || value === '0');
    if (d._type === 'undefined') return undefined;
    if (d._type === 'null') return null;
    if (d._type === 'any') return value;
    if (d._type === 'nat') return Number(value);
    if (d._type === 'int') return Number(value);
    if (d._type === 'float') return Number(value);
    return absurd(d._type);
  }
    
  if (d instanceof t.OneOf) {
    throw new Error('[config-reader] OneOf is not supported');
  }
  
  if (d instanceof t.Discriminate) {
    try {
      return JSON.parse(value);
    } catch(e) {
      return value;
    }
  }
  
  if (d instanceof t.Pure) {
    return value;
  }

  if (d instanceof t.ToDecoder) {
    if (d instanceof WithDefault) {
      return fromString(d._decoder, value);      
    }
    if (d instanceof t.Variants) {
      return value;
    }    
    return fromString(d.toDecoder(), value);
  }

  if (d instanceof t.Custom) {
    throw new Error('[config-reader]: CustomDecoder is not supported');
  }

  if (d instanceof t.AtDecoder) {
    throw new Error('[config-reader]: AtDecoder is not supported');
  }

  if (d instanceof t.Chain) {
    throw new Error('[config-reader]: ChainDecoder is not supported');
  }

  return absurd(d);
}


export class ConfigDescription<A> extends t.ToDecoder<A> {
  constructor(
    readonly _decoder: Decoder<A>,
    readonly _description: string,    
  ) { super(); }

  toDecoder() {
    return this._decoder;
  }
}

// Информация о cli параметрe
export type InfoItem = { type: string; default?: unknown; description?: string };


export function configInfo(decoder: RecordDecoder<any>, prefix=''): Record<string, InfoItem> {
  return Object.keys(decoder._description).reduce((acc, k) => (acc[k] = go(decoder._description[k]), acc), {});

  function go(d: Decoder<any>): InfoItem {
    if (d instanceof t.ArrayDecoder) {
      const { type } = go(d._decoder);
      return { type: 'Array<' + type + '>' };
    }

    if (d instanceof t.Dict) {
      const { type } = go(d._decoder);
      return { type: 'Record<string, ' + type + '>' };
    }
    
    if (d instanceof t.RecordDecoder) {
      // TODO: печать полей
      return { type: 'Record<string, unknown>' };
    }

    if (d instanceof t.Primitive) {
      if (d._type === 'string') return { type: 'string' };
      if (d._type === 'boolean') return { type: 'boolean' };
      if (d._type === 'undefined') return { type: 'undefined' };
      if (d._type === 'null') return { type: 'null' };
      if (d._type === 'any') return { type: 'any' };
      if (d._type === 'nat') return { type: 'number' };
      if (d._type === 'int') return { type: 'number' };
      if (d._type === 'float') return { type: 'number' };
      return absurd(d._type);
    }
    
    if (d instanceof t.OneOf) {
      return { type: d._alternatives.map(go).map(x => x.type).join('|') };
    }
    
    if (d instanceof t.Discriminate) {
      // TODO: печать дискриминаторов
      return { type: Object.keys(d._alternatives).map(k => go(d._alternatives[k]).type).join('|') };
    }
    
    if (d instanceof t.Pure) {
      return { type: JSON.stringify(d._value) };
    }

    if (d instanceof t.ToDecoder) {
      if (d instanceof WithDefault) {
        const info = go(d._decoder);
        info.default = d._default;
        return info;
      }

      if (d instanceof t.Variants) {
        return { type: d._variants.map(x => typeof(x) === 'string' ? x : JSON.stringify(x)).join('|') };
      }
      
      const info = go(d.toDecoder());
      if (d instanceof ConfigDescription) {
        info.description = d._description;
      }
      return info;
    }

    if (d instanceof t.Custom) {
      throw new Error('[config-reader]: CustomDecoder is not supported');
    }

    if (d instanceof t.AtDecoder) {
      throw new Error('[config-reader]: AtDecoder is not supported');
    }

    if (d instanceof t.Chain) {
      throw new Error('[config-reader]: ChainDecoder is not supported');
    }

    return absurd(d);
  }
}


declare module "../decoder" {
  interface DecoderBase<A> {
    withDescription(description: string): Decoder<A>;
    validateConfigs(...sources: ConfigSource[]): Either<Problem, A>;
  }
}

t.DecoderBase.prototype.withDescription = function(description) {
  return new ConfigDescription(this as any, description);
};

t.DecoderBase.prototype.validateConfigs = function(...sources) {
  return this.validate(merge(this as any, ...sources));
};
