import * as React from 'react';
import { isEqual } from 'lodash';
import memoize from 'memoize-one';
import { Chip, Checkbox, MenuItem, ListItemText, StandardProps } from '@material-ui/core';
import * as classNames from 'classnames';
import withStyles, { StyleRules } from '@material-ui/core/styles/withStyles';
import { Theme } from '@material-ui/core/styles/createMuiTheme';
import AutoComplete from './AutoComplete';
import { Props as TextFieldProps } from '../fields/TextField';
import { Source } from '../fields/AutoComplete/state-machine';
import { FieldProps } from '../create-form';


// Props
export type Props<A extends any[]> = StandardProps<React.HTMLProps<HTMLDivElement>, string, 'value'|'disabled'> & FieldProps<A> & {
  source: Source<A[number]>;
  fullWidth?: boolean;
  textFieldProps?: Partial<TextFieldProps>;
  renderItem?(a: A[number], selected: boolean): React.ReactNode;
  placeholder?: string;
  printItem?(a: A[number]): string;
  id?: string; // Используется для установки `key` пропсов
  isEqual?(a: A[number], b: A[number]): boolean;
}


// @ts-ignore Component
@withStyles(styles)
export default class MultiSelect<A extends any[]> extends React.Component<Props<A>> {
  static defaultProps = {
    printItem: String,
  } as any;
  
  handleDelete = memoize((idx: number) => (e?: React.MouseEvent<HTMLDivElement>) => {
    const { value, onValueChange, disabled } = this.props; if (disabled) return;
    // @ts-ignore
    const nextValue = value!.slice();
    nextValue.splice(idx, 1);
    if (e) {
      const nextSibling = e.currentTarget.parentNode!.childNodes[idx + 1] as HTMLElement|null;
      nextSibling && nextSibling.focus();
    }
    onValueChange && onValueChange(nextValue);
  });

  isEqual(a: A, b: A) {
    return (this.props.isEqual || isEqual)(a, b);
  }

  handleValueChange = (item: A) => {
    const { value, onValueChange } = this.props;;
    // @ts-ignore
    const nextValue = value!.slice();

    if (!item) {
      onValueChange && onValueChange([]);
      return;
    }
    
    for (let i = 0; i < nextValue.length; i++) {
      if (this.isEqual(nextValue[i], item)) {
        nextValue.splice(i, 1);
        onValueChange && onValueChange(nextValue);
        return;
      }
    }
    nextValue.push(item);
    onValueChange && onValueChange(nextValue);
  };

  renderChips = memoize((value: Props<A>['value']) => {
    const { classes, printItem, disabled } = this.props;
    const id = this.props.id || 'id';
    const chipClassName = classNames({ [classes!.disabled!]: disabled });
    // @ts-ignore
    return !value.length ? null : <div><span className={classes!.chips}>
      {(value as any).map((ch, i) => (
        <Chip key={ch[id]} className={chipClassName} label={printItem!(ch)} onClick={this.handleDelete(i)} onDelete={this.handleDelete(i)}/>
      ))}
    </span></div>;
  });

  renderSuggestion = (item: A) => {
    const { renderItem, printItem, value } = this.props;
    // @ts-ignore
    const selected = !!value!.find(x => this.isEqual(x, item));
    const id = this.props.id || 'id';

    if (renderItem) return renderItem(item, selected);

    return <MenuItem key={item[id]} disableRipple>
      <Checkbox disableRipple checked={selected}/>
      <ListItemText primary={(printItem || String)(item)}/>
    </MenuItem>;
  };

  render() {
    const { ctx, classes, fullWidth, className, source, renderItem, printItem, id, textFieldProps, value, error, disabled, onValueChange, placeholder, ...rest } = this.props;
    const rootClass = classNames(className, classes!.root, {
      [classes!.disabled!]: disabled,
      [classes!.fullWidth!]: fullWidth,
    });
    
    return <div {...rest} className={rootClass}>
      {this.renderChips(value)}
      <AutoComplete<A|null>
        ctx={ctx}
        value={null}
        printItem={printItem}
        source={source}
        // @ts-ignore
        renderItem={this.renderSuggestion}
        onValueChange={this.handleValueChange}
        textFieldProps={textFieldProps}
        placeholder={placeholder}
        error={error}
        disabled={disabled}
        fullWidth={fullWidth}
        keepOpenAfterSelect
        openOnClick
        openOnFocus
        nonNull
        />
    </div>;
  }
}


// Style
export function styles(theme: Theme): StyleRules {
  const { unit } = theme.spacing;
  
  return {
    root: {
    },

    chips: {
      marginLeft: -unit,
      marginBottom: unit,
      display: 'inline-block',
      '& > *': {
        display: 'inline-flex',
        marginLeft: unit,
        marginTop: unit,
      },
    },
    
    disabled: {
      cursor: 'not-allowed',
    },
    
    fullWidth: {
      width: '100%',
    },    
  };
}
