import { Either } from '../either';
import { Expr, absurd } from '../types';


// This library helps you declaratively construct type-safe validators for untyped JSON
// inspired by http://package.elm-lang.org/packages/elm-lang/core/3.0.0/Json-Decode
// and https://github.com/gcanti/io-ts


/** ADT */
export type Decoder<A> =
  | CustomDecoder<A>
  | ArrayDecoder<A>
  | DictionaryDecoder<A>
  | RecordDecoder<A>
  | AtDecoder<A>
  | PrimitiveDecoder<A>
  | PureDecoder<A>
  | ChainDecoder<A>
  | OneOfDecoder<A>
  | DiscriminatedDecoder<A>
  | HasDecoder<A>


/**
 * Информация о возникшей ошибке
 */
export type Problem = {
  value: any;
  rootValue: any;
  decoder: Decoder<any>;
  rootDecoder: Decoder<any>;
  message: string;
}


/**
 * Опции для `DecoderBase.prototype.validate`
 */
export interface ValidateOptions {
  printProblem: boolean;
}

const defaultValidateOptions: ValidateOptions = {
  printProblem: false,
};


// Базовый класс для наследования методов
export class DecoderBase<A> {
  readonly _A: A;
  static rootDecoder: Decoder<any>|undefined = undefined;
  static rootValue: any = undefined;

  /**
   * Печать декодера в виде выражения которым он был создан
   */
  prettyPrint(): string {
    const self = this as any as Decoder<A>;
    return prettyPrint(self);
  }

  /**
   * Валидация произвольного значения
   */
  validate(value: any, options_?: Partial<ValidateOptions>): Either<Problem, A> {
    const self = this as any as Decoder<A>;
    let output: Either<Problem, A>;
    
    if (DecoderBase.rootValue === undefined) {
      DecoderBase.rootValue = value;
      DecoderBase.rootDecoder = self;
      output = doValidate(self, value)
      DecoderBase.rootValue = undefined;
      DecoderBase.rootDecoder = undefined;
    } else {
      output = doValidate(self, value)
    }

    // Печать проблемы в консоль если выставлена соответствующая опция
    if (output.tag === 'Left') {
      const options = { ...defaultValidateOptions, ...options_ };
      if (options.printProblem) {
        printProblems(output.value);
      }
    }
    return output;
  }

  map<B>(f: (a: A) => B): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, x => of(f(x))));
  }

  mapTo<B>(value: B): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, () => of(value)));
  }

  chain<B>(f: (a: A) => Decoder<B>): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, f));
  }
  
  chainTo<B>(value: Decoder<B>): Decoder<B> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(fail, () => value));
  }
  
  mapProblem(f: (a: Problem) => Problem): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(f, x => x));
  }
  
  mapProblemTo(value: Problem): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, x => x.fold(() => value, x => x));
  }

  withDefault(defValue: A): Decoder<A>;
  withDefault<B extends Expr>(defValue: B): Decoder<A|B>;
  withDefault<B extends Expr>(defValue: B): Decoder<A|B> {
    return new WithDefaultDecoder(this as any, defValue);
  }

  refine(pred: (a: A) => boolean): Decoder<A> {
    const self = this as any as Decoder<A>;
    return new ChainDecoder(self, ethr => {
      switch (ethr.tag) {
        case 'Left': return new PureDecoder(ethr);
        case 'Right': return pred(ethr.value) ? new PureDecoder(ethr) : fail('refinement failed');
      }
    });
  }
}


/**
 * Декодер с произвольной функцией для валидации
 */
export class CustomDecoder<A> extends DecoderBase<A> {
  readonly tag: 'CustomDecoder' = 'CustomDecoder';

  constructor(
    readonly name: string,
    readonly validateCustom: (val: any) => Either<Problem|string, A>,
  ) { super(); }
}


/** 
 * Декодер массивов
 */
export class ArrayDecoder<A> extends DecoderBase<A> {
  readonly tag: 'ArrayDecoder' = 'ArrayDecoder';

  constructor(
    readonly decoder: Decoder<any>,
  ) { super(); }
}


/** 
 * Декодер словарей (или хешей, те объектов с произвольным кол-вом ключей)
 */
export class DictionaryDecoder<A> extends DecoderBase<A> {
  readonly tag: 'DictionaryDecoder' = 'DictionaryDecoder';

  constructor(
    readonly decoder: Decoder<any>,
  ) { super(); }
}


/** 
 * Декодер записей (объекты с фиксированным количеством полей)
 */
export class RecordDecoder<A> extends DecoderBase<A> {
  readonly tag: 'RecordDecoder' = 'RecordDecoder';

  constructor(
    readonly description: Record<string, Decoder<any>>,
  ) { super(); }
}


/**
 * Декодер поля с указанным индексом
 */
export class AtDecoder<A> extends DecoderBase<A> {
  readonly tag: 'AtDecoder' = 'AtDecoder';

  constructor(
    readonly path: Array<string|number>,
    readonly decoder: Decoder<A>,
  ) { super(); }
}


/**
 * Декодер для примитовов
 */
export class PrimitiveDecoder<A> extends DecoderBase<A> {
  readonly tag: 'PrimitiveDecoder' = 'PrimitiveDecoder';

  constructor(
    readonly primitive: 'null'|'undefined'|'string'|'boolean'|'any'|'nat'|'int'|'float'
  ) { super(); }
}


/** 
 * Тривиальный декодер
 */
export class PureDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly value: Either<Problem|string, A>,
  ) { super(); }
}


/**
 * Монадный комбинатор
 */
export class ChainDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly decoder: Decoder<any>,
    readonly andThen: (x: Validation<any>) => Decoder<A>,
  ) { super(); }
}


/** 
 * `oneOf` комбинатор
 */
export class OneOfDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly alternatives: Decoder<any>[],
  ) { super(); }
}


/** 
 * `discriminateOn` комбинатор
 */
export class DiscriminatedDecoder<A> extends DecoderBase<A> {
  constructor(
    readonly discriminator: string|number,
    readonly alternatives: Record<string|number, Decoder<any>>,
  ) { super(); }
}


/** 
 * `discriminateOn` комбинатор
 */
export abstract class HasDecoder<A> extends DecoderBase<A> {
  abstract toDecoder(): Decoder<A>;
}


/** 
 * `discriminateOn` комбинатор
 */
export class WithDefaultDecoder<A> extends HasDecoder<A> {
  constructor(
    readonly decoder: Decoder<A>,
    readonly defaultValue: A,
  ) { super(); }

  toDecoder() {
    return new OneOfDecoder<A>([this.decoder, of(this.defaultValue)]);
  }
}


/** 
 * Тип результата для функции-валидатора
 */
export type Validation<A> = Either<Problem|string, A>;


/**
 * Конструктир кастомных декодеров
 */
export function decoder<A>(validate: (value: any) => Either<Problem|string, A>): CustomDecoder<A>;
export function decoder<A>(name: string, validate: (value: any) => Either<Problem|string, A>): CustomDecoder<A>;
export function decoder(): any {
  if (arguments.length === 1) return new CustomDecoder('custom', arguments[0]);
  if (arguments.length === 2) return new CustomDecoder(arguments[0], arguments[1]);
  throw new TypeError(`decoder: invalid number of arguments`);
}


/**
 * Алиас для `x => new PureDecoder(Either.of(x))`
 */
export function of<A extends Expr>(a: A): PureDecoder<A> {
  return new PureDecoder(Either.of(a));
}


/**
 * Алиас для `x => new PureDecoder(Either.failure(x))`
 */
export function fail(x: Problem|string): Decoder<never> {
  return new PureDecoder(Either.failure(x));
}


/** 
 * Аппликативный комбинатор
 * TODO: сделать отдельным вариантом в ADT для реализации `prettyPrint`
 */
export function ap<A,B>(a: Decoder<A>, f:(a: A) => B): CustomDecoder<B>;
export function ap<A,B,C>(a: Decoder<A>, b: Decoder<B>, f:(a: A, b: B) => C):CustomDecoder<C>;
export function ap<A,B,C,D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, f:(a: A, b: B, c: C) => D):CustomDecoder<D>;
export function ap<A,B,C,D,E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, f:(a: A, b: B, c: C, d: D) => E):CustomDecoder<E>;
export function ap<A,B,C,D,E,F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f:(a: A, b: B, c: C, d: D, e: E) => F):CustomDecoder<F>;
export function ap<A,B,C,D,E,F,G>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, f:(a: A, b: B, c: C, d: D, e: E, f: F) => G):CustomDecoder<G>;
export function ap<A,B,C,D,E,F,G,H>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G) => H):CustomDecoder<H>;
export function ap<A,B,C,D,E,F,G,H,J>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H) => J):CustomDecoder<J>;
export function ap<A,B,C,D,E,F,G,H,J,K>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, j: Decoder<J>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H, j: J) => K):CustomDecoder<K>;
export function ap<A,B,C,D,E,F,G,H,J,K,L>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f_:Decoder<F>, g: Decoder<G>, h: Decoder<H>, j: Decoder<J>, k: Decoder<K>, f:(a: A, b: B, c: C, d: D, e: E, f: F, g: G, h: H, j: J, k: K) => L):CustomDecoder<L>;
export function ap(...args: Array<Decoder<any> | Function>): Decoder<any> {
  return decoder('ap', (val) => {
    const func = args[args.length - 1] as Function;
    const results: Array<any> = [];
    for (let i = 0; i < args.length - 1; i++) {
      const dec = args[i] as Decoder<any>;
      const ethr = dec.validate(val);
      switch(ethr.tag) {
        case 'Left': return ethr;
        case 'Right': results.push(ethr.value); break;
      }
    }
    return Either.of(func.apply(undefined, results));
  });
}


// Примитивы
const anyDecoder = new PrimitiveDecoder<any>('any');
const stringDecoder = new PrimitiveDecoder<string>('string');
const booleanDecoder = new PrimitiveDecoder<boolean>('boolean');
const nullDecoder = new PrimitiveDecoder<null>('null');
const undefinedDecoder = new PrimitiveDecoder<undefined>('undefined');
export const nat = new PrimitiveDecoder<number>('nat');
export const int = new PrimitiveDecoder<number>('int');
export const float = new PrimitiveDecoder<number>('float');


// Экспорт с переименованием
export { anyDecoder as any, stringDecoder as string, booleanDecoder as boolean, nullDecoder as null, undefinedDecoder as undefined };


/**
 * Сопоставление с несколькими декодерами до первого успешного
 * сравнвния
 */
export function oneOf<A>(a: Decoder<A>): OneOfDecoder<A>;
export function oneOf<A,B>(a: Decoder<A>, b: Decoder<B>): OneOfDecoder<A|B>;
export function oneOf<A,B,C>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>): OneOfDecoder<A|B|C>;
export function oneOf<A,B,C,D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>): OneOfDecoder<A|B|C|D>;
export function oneOf<A,B,C,D,E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>): OneOfDecoder<A|B|C|D|E>;
export function oneOf<A,B,C,D,E,F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>): OneOfDecoder<A|B|C|D|E|F>;
export function oneOf<array extends Decoder<any>[]>(array: array): Decoder<array[number]['_A']>;
export function oneOf(): Decoder<any> {
  const decoders = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  return new OneOfDecoder(decoders);
}


export function array<A>(decoder: Decoder<A>): Decoder<A[]> {
  return new ArrayDecoder(decoder);
}

export function record<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}

// Вложенные записи в декодерах
export function record2<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}

// Вложенные записи в декодерах
export function record3<fields extends { [k: string]: Decoder<any> }>(fields: fields): RecordDecoder<{[k in keyof fields]: fields[k]['_A'] }> {
  return new RecordDecoder(fields);
}


export const d = array(record({ a: array(record({ b: record({ c: stringDecoder }) })) } ));

export function dict<A>(decoder: Decoder<A>): Decoder<Record<string, A>> {
  return new DictionaryDecoder(decoder);
}

export function at<A>(path: string|string[]|number|number[], decoder: Decoder<A>): Decoder<A> {
  return new AtDecoder(Array.isArray(path) ? path : [path], decoder);
}

export const date: Decoder<Date> = decoder('date', (value) => {
  if (typeof (value) !== 'string') return Either.failure('not a string');
  const d = new Date(value);
  return isNaN(d.getTime()) ? Either.failure('error parsing date from string') : Either.of(d);
});


/** 
 * Хелпер для объявления декодеров в стиле Elm. Используется вместе в
 * `ap`
 */
export function required<A>(key: string|string[], dec: Decoder<A>): Decoder<A> {
  return at(key, dec);
}


/**
 * Декодер для дискриминированных объединений
 * ```ts
 * const roleDecoder = t.discrinimate('type', {
 *  'director': record({ name: t.string, private_info: t.string }),
 *  'employee': record({ name: t.string }),
 * });
 * // => Decoder<{ { type: 'director', ... } | { type: 'employee', ... }}>
 * ```
 */
export type DiscriminateOn<TagKey extends string, Descriptor extends Record<string, Decoder<any>>> = DiscriminatedDecoder<{ [K in keyof Descriptor]: { [K2 in TagKey]: K } & Descriptor[K]['_A']}[keyof Descriptor]>;
export function discriminate<TagKey extends string, Descriptor extends Record<string, Decoder<any>>>(key: TagKey, record: Descriptor): DiscriminateOn<TagKey, Descriptor> {
  return new DiscriminatedDecoder(key, record);
}


/** 
 * Хелпер для объявления декодеров в стиле Elm. Используется вместе в
 * `ap`
 */
export function optional<A>(key: string|string[], dec: Decoder<A>, def: A): Decoder<A>;
export function optional<A,B>(key: string|string[], dec: Decoder<A>, def: B): Decoder<A|B>;
export function optional(key: string|string[], dec: Decoder<any>, def: any): Decoder<any> {
  return required(key, dec).withDefault(def);
}


/** 
 * Создание декодера перечислением всех допустимых значений 
 */
export function literals<A extends Expr[]>(...array: A): Decoder<A[number]>;
export function literals<A extends Expr[]>(array: A): Decoder<A[number]>;
export function literals(): Decoder<any> {
  const literals: Expr[] = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  return new OneOfDecoder(literals.map(x => decoder(v => v === x ? Either.of(v) : Either.failure(`expected ${x}, got ${fancyTypeOf(v)}`))));
}


/**
 * Кортежи разных размеров. Проверяемое значение необязательно должно
 * быть массивом
 * ```ts
 *   const pair = t.tuple(t.string, t.number);
 *   const pair_2 = t.record({ '0': t.string, '1': t.number }); // тоже самое
 * ```
 */
export function tuple<A>(a: Decoder<A>): Decoder<[A]>;
export function tuple<A, B>(a: Decoder<A>, b: Decoder<B>): Decoder<[A, B]>;
export function tuple<A, B, C>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>): Decoder<[A, B, C]>;
export function tuple<A, B, C, D>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>): Decoder<[A, B, C, D]>;
export function tuple<A, B, C, D, E>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>): Decoder<[A, B, C, D, E]>;
export function tuple<A, B, C, D, E, F>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>): Decoder<[A, B, C, D, E, F]>;
export function tuple<A, B, C, D, E, F, G>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>): Decoder<[A, B, C, D, E, F, G]>;
export function tuple<A, B, C, D, E, F, G, H>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>): Decoder<[A, B, C, D, E, F, G, H]>;
export function tuple<A, B, C, D, E, F, G, H, I>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>, i: Decoder<I>): Decoder<[A, B, C, D, E, F, G, H, I]>;
export function tuple<A, B, C, D, E, F, G, H, I, J>(a: Decoder<A>, b: Decoder<B>, c: Decoder<C>, d: Decoder<D>, e: Decoder<E>, f: Decoder<F>, g: Decoder<G>, h: Decoder<H>, i: Decoder<I>, j: Decoder<J>): Decoder<[A, B, C, D, E, F, G, H, I, J]>;
export function tuple<A>(args: Decoder<A>[]): Decoder<A[]>;
export function tuple(): Decoder<any> {
  const args: Decoder<any>[] = Array.isArray(arguments[0]) ? arguments[0] : Array.prototype.slice.call(arguments);
  // @ts-ignore
  return ap(...args.map((decoder, idx) => at(idx, decoder)), (...xs) => xs);
}


/** 
 * Печать проблемы с консоль
 */
export function printProblems(problems: Problem): void {
  console.log(`%cValidation error: ${problems.message}`, 'color: #f00; font-size: 16px; font-weight: 600;')
  console.log('%cfailed decoder:', 'font-weight: 600;', problems.decoder.prettyPrint());
  console.log('%croot decoder:', 'font-weight: 600;', problems.rootDecoder.prettyPrint());
  console.log('%cfailed value:', 'font-weight: 600;', problems.value);
  console.log('%croot value:', 'font-weight: 600;', problems.rootValue);
  //console.log('decoders ', problems.decoders.map(x => x.prettyPrint()));
}


// Хелпер
function fancyTypeOf(value: any): string {
  return Object.prototype.toString.call(value);
}


/**
 * Выполнение валидации
 */
export function doValidate<A>(decoder: Decoder<A>, value: unknown): Either<Problem, A> {
  if (decoder instanceof CustomDecoder) {
    return decoder.validateCustom(value).mapLeft(projectProblem);
  }
  
  if (decoder instanceof ArrayDecoder) {
    const output: any[] = [];
    if (!Array.isArray(value)) return Either.failure(projectProblem('not an array'));
    for (let i = 0; i < value.length; i++) {
      const ethr = doValidate(decoder.decoder, value[i]);
      switch(ethr.tag) {
        case 'Left': { return ethr; }
        case 'Right': output.push(ethr.value); break;
      }
    }
    return Either.of(output as any as A);
  }

  if (decoder instanceof DictionaryDecoder) {
    if (typeof (value) !== 'object') return Either.failure(projectProblem('not an object'));
    if (value === null) return Either.failure(projectProblem('found null'));
    const output: { [k: string]: A } = {};
    for (let key in value) {
      if (!value.hasOwnProperty(key)) continue;
      const ethr = doValidate(decoder.decoder, value[key]);
      switch(ethr.tag) {
        case 'Left': { return ethr; }
        case 'Right': output[key] = ethr.value; break;
      }
    }
    return Either.of(output as any as A);
  }

  if (decoder instanceof RecordDecoder) {
    if (typeof (value) !== 'object') return Either.failure(projectProblem('not an object'));
    if (value === null) return Either.failure(projectProblem('found null'));
    const output: { [k: string]: any } = {};
    for (let key in decoder.description) {
      if (!decoder.description.hasOwnProperty(key)) continue;
      const ethr = doValidate(decoder.description[key], value[key]);
      switch(ethr.tag) {
        case 'Left': { return ethr; }
        case 'Right': output[key] = ethr.value; break;
      }
    }
    return Either.of(output as any as A);
  }

  if (decoder instanceof AtDecoder) {
    let iter = value as any;
    for (let i in decoder.path) {
      if (iter === undefined || !iter.hasOwnProperty(decoder.path[i])) {
        iter = undefined;
        break;
      }
      iter = iter[decoder.path[i]];
    }
    return doValidate(decoder.decoder, iter);
  }
 
  if (decoder instanceof PrimitiveDecoder) {
    switch (decoder.primitive) {
      case 'null': return value === null ? Either.of(value as A) : Either.failure(projectProblem(`expected null, got ${fancyTypeOf(value)}`));
      case 'undefined': return value === undefined ? Either.of(value as A) : Either.failure(projectProblem(`expected undefined, got ${fancyTypeOf(value)}`));
      case 'string': return typeof(value) === 'string' ? Either.of(value as any) : Either.failure(projectProblem(`expected a string, got ${fancyTypeOf(value)}`));
      case 'boolean': return typeof(value) === 'boolean' ? Either.of(value as any) : Either.failure(projectProblem(`expected a boolean, got ${fancyTypeOf(value)}`));
      case 'any': return Either.of(value as A);
      case 'nat': return typeof (value) !== 'number' ? Either.failure(projectProblem('not a number')) : (value|0) === value && value >= 0 ? Either.of(value as any) : Either.failure(projectProblem('not a natural number'));
      case 'int': return typeof (value) !== 'number' ? Either.failure(projectProblem('not a number')) : (value|0) === value ? Either.of(value as any) : Either.failure(projectProblem('not an integer'))
      case 'float': return typeof (value) !== 'number' ? Either.failure(projectProblem('not a number')) : Either.of(value as any);
    }
    return absurd(decoder.primitive);
  }

  if (decoder instanceof PureDecoder) {
    return decoder.value.mapLeft(projectProblem);
  }
  
  if (decoder instanceof OneOfDecoder) {
    for (const d of decoder.alternatives) {
      const ethr = doValidate(d, value);
      switch(ethr.tag) {
        case 'Left': break;
        case 'Right': return ethr;
      }
    }
    return Either.failure(projectProblem('none of decoders succeded'));
  }

  if (decoder instanceof ChainDecoder) {
    return doValidate(decoder.andThen(doValidate(decoder.decoder, value)), value);
  }

  if (decoder instanceof DiscriminatedDecoder) {
    const key = decoder.discriminator;
    const record = decoder.alternatives;
    if (typeof(value) !== 'object' || !value) return Either.failure(projectProblem('expected an object'));
    if (!value.hasOwnProperty(key)) return Either.failure(projectProblem(`expected input would have key '${key}'`));
    if (!(value[key] in record)) return Either.failure(projectProblem(`unknown value of discriminated key ${value[key]}`));
    return record[value[key]].validate(value).map(x => ({ ...x, [key]: value[key] }));
  }

  if (decoder instanceof HasDecoder) {
    return doValidate(decoder.toDecoder(), value);
  }

  return absurd(decoder);

  function projectProblem(problem: Problem|string): Problem {
    if (typeof(problem) !== 'string') return problem;
    return { value, rootValue: DecoderBase.rootValue, decoder, rootDecoder: DecoderBase.rootDecoder || decoder, message: problem, };
  }  
}



/**
 * Печать декодера.
 * ```ts
 * const user = t.record({ name: t.string });
 * console.log(t.prettyPrint(user)); // => "t.record({ name: t.string })"
 * ```
 */
export function prettyPrint(decoder: Decoder<any>): string {
  if (decoder instanceof CustomDecoder) return `t.decoder(${JSON.stringify(decoder.name)}, <func>)`;
  if (decoder instanceof ArrayDecoder) return `t.array(${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof DictionaryDecoder) return `t.dict(${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof RecordDecoder) return `t.record({ ${Object.keys(decoder.description).map(k => decoder.description[k].prettyPrint()).join(', ')} })`;
  if (decoder instanceof AtDecoder) return `t.at(${JSON.stringify(decoder.path)}, ${decoder.decoder.prettyPrint()})`;
  if (decoder instanceof PrimitiveDecoder) return `t.${decoder.primitive}`;
  if (decoder instanceof PureDecoder) return `t.of(${JSON.stringify(decoder.value)})`;
  if (decoder instanceof ChainDecoder) return `${decoder.prettyPrint()}.chain(<func>)`;
  if (decoder instanceof OneOfDecoder) return `t.oneOf(${decoder.alternatives.map(x => x.prettyPrint()).join(', ')})`;
  if (decoder instanceof DiscriminatedDecoder) {
    const discriminator = JSON.stringify(decoder.discriminator);
    const alternatives = Object.keys(decoder.alternatives).map(k => JSON.stringify(k) + ": " + decoder.alternatives[k].prettyPrint()).join(', ');
    return `t.discriminateOn(${JSON.stringify(discriminator)}, ${alternatives})`;
  }
  if (decoder instanceof HasDecoder) {
    return prettyPrint(decoder.toDecoder());
  }
  return absurd(decoder);
}

